<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Product; // Import the Product model
use Inertia\Inertia; // Import Inertia
use App\Http\Requests\ProductStoreRequest; // Import the ProductStoreRequest
use App\Http\Requests\ProductUpdateRequest; // Import the ProductUpdateRequest
use App\Models\ProductCategory;
use Illuminate\Support\Str; // Import Str for UUID generation
use Illuminate\Support\Facades\Storage; // Import Storage for file operations

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $products = Product::latest()->paginate(10); // Fetch products with pagination
        return Inertia::render('Admin/Products/Index', [
            'products' => $products,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = ProductCategory::all(); // Fetch all product categories
        return Inertia::render('Admin/Products/Create', [
            'categories' => $categories,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ProductStoreRequest $request)
    {
        try {
            $validated = $request->validated();

            $product = new Product();
            $product->id = Str::uuid(); // Generate UUID for the product
            $product->external_id = Str::uuid(); // Generate UUID for external_id
            $product->name = $validated['name'];
            $product->description = $validated['description'] ?? null;
            $product->price = $validated['price'];
            $product->stock = $validated['stock'];
            $product->category = $validated['category'] ?? null;

            $imagePaths = [];
            $imageUrls = [];
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $path = $image->store('products', 'public');
                    $imagePaths[] = $path;
                    $imageUrls[] = url($path);
                }
            }
            $product->images = $imagePaths;
            $product->image_urls = $imageUrls;

            $product->save();

            return redirect()->route('admin.products.index')->with('success', 'Product created successfully.');
        } catch (\Throwable $th) {
            dd($th->getMessage());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Product $product)
    {
        return Inertia::render('Admin/Products/Show', [
            'product' => $product,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        $categories = ProductCategory::all(); // Fetch all product categories
        return Inertia::render('Admin/Products/Edit', [
            'product' => $product,
            'categories' => $categories,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ProductUpdateRequest $request, Product $product)
    {
        $validated = $request->validated();

        $product->name = $validated['name'];
        $product->description = $validated['description'] ?? null;
        $product->price = $validated['price'];
        $product->stock = $validated['stock'];
        $product->category = $validated['category'] ?? null;

        if ($request->hasFile('images')) {
            // Delete old images if new ones are uploaded
            foreach ($product->images as $oldImage) {
                Storage::disk('public')->delete($oldImage);
            }

            $imagePaths = [];
            $imageUrls = [];
            foreach ($request->file('images') as $image) {
                $path = $image->store('products', 'public');
                $imagePaths[] = $path;
                $imageUrls[] = url($path);
            }
            $product->images = $imagePaths;
            $product->image_urls = $imageUrls;
        }

        $product->save();

        return redirect()->route('admin.products.index')->with('success', 'Product updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        // Delete associated images from storage
        if ($product->images) {
            foreach ($product->images as $imagePath) {
                Storage::disk('public')->delete(str_replace('/storage/', '', $imagePath));
            }
        }

        $product->delete();

        return redirect()->route('admin.products.index')->with('success', 'Product deleted successfully.');
    }
}