<?php

namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use Illuminate\Support\Str; // For UUID generation

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (request()->wantsJson()) {
            $products = Product::orderBy('name')->get();
            return response()->json($products);
        }
        return Inertia::render('Admin/Products/Index', [
            'products' => Product::orderBy('name')->get(),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/Products/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048', // Assuming single image upload for now
            'category' => 'nullable|string|max:255',
        ]);

        $imagePaths = [];
        $imageUrls = [];

        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('products', 'public');
            $imageUrl = Storage::disk('public')->url($imagePath);
            $imagePaths[] = $imagePath;
            $imageUrls[] = $imageUrl;
        }

        $product = Product::create([
            'id' => Str::uuid(), // Generate UUID for 'id'
            'external_id' => Str::uuid(), // Generate UUID for 'external_id'
            'name' => $request->name,
            'description' => $request->description,
            'price' => $request->price,
            'stock' => $request->stock,
            'images' => json_encode(['paths' => $imagePaths, 'urls' => $imageUrls]), // Store as JSON
            'category' => $request->category,
        ]);

        if (request()->wantsJson()) {
            return response()->json($product, 201);
        }
        return redirect()->route('admin.products.index')->with('success', 'Product created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $product = Product::findOrFail($id);
        return response()->json($product);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        return Inertia::render('Admin/Products/Edit', [
            'product' => $product,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048', // Assuming single image upload for now
            'category' => 'nullable|string|max:255',
        ]);

        $imagePaths = $product->images['paths'] ?? [];
        $imageUrls = $product->images['urls'] ?? [];

        if ($request->hasFile('image')) {
            // Delete old image if exists
            if (!empty($imagePaths)) {
                foreach ($imagePaths as $oldImagePath) {
                    Storage::disk('public')->delete($oldImagePath);
                }
            }
            $imagePath = $request->file('image')->store('products', 'public');
            $imageUrl = Storage::disk('public')->url($imagePath);
            $imagePaths = [$imagePath]; // Replace with new image
            $imageUrls = [$imageUrl]; // Replace with new image
        }

        $product->update([
            'name' => $request->name,
            'description' => $request->description,
            'price' => $request->price,
            'stock' => $request->stock,
            'images' => json_encode(['paths' => $imagePaths, 'urls' => $imageUrls]),
            'category' => $request->category,
        ]);

        if (request()->wantsJson()) {
            return response()->json($product);
        }
        return redirect()->route('admin.products.index')->with('success', 'Product updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        // Delete associated images
        if ($product->images) {
            $images = json_decode($product->images, true);
            if (isset($images['paths'])) {
                foreach ($images['paths'] as $imagePath) {
                    Storage::disk('public')->delete($imagePath);
                }
            }
        }
        $product->delete();

        if (request()->wantsJson()) {
            return response()->json(null, 204);
        }
        return redirect()->route('admin.products.index')->with('success', 'Product deleted successfully.');
    }
}