<?php

namespace App\Http\Controllers;

use App\Models\Service;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use App\Http\Requests\ServiceStoreUpdateRequest; // Import the new request

class ServiceController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (request()->wantsJson()) {
            $services = Service::orderBy('order')->get();
            return response()->json($services);
        }
        return Inertia::render('Admin/Services/Index', [
            'services' => Service::orderBy('order')->get(),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/Services/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ServiceStoreUpdateRequest $request) // Use the new request
    {
        $validated = $request->validated();

        $imagePath = null;
        $imageUrl = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('services', 'public');
            $imageUrl= url($imagePath);
        }
        
        $service = Service::create([
            'title' => $validated['title'],
            'description' => $validated['description'],
            'icon' => $validated['icon'] ?? null,
            'order' => $validated['order'],
            'image_path' => $imagePath,
            'image_url' => $imageUrl,
            'type' => $validated['type'],
            'price' => $validated['price'],
            'is_visible' => $validated['is_visible'] ?? true,
        ]);

        if (request()->wantsJson()) {
            return response()->json($service, 201);
        }
        return redirect()->route('admin.services.index')->with('flash.success', 'Service created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $service = Service::findOrFail($id);
        return response()->json($service);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Service $service)
    {
        return Inertia::render('Admin/Services/Edit', [
            'service' => $service,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ServiceStoreUpdateRequest $request, Service $service) // Use the new request
    {
        $validated = $request->validated();

        $imagePath = $service->image_path;
        $imageUrl = $service->image_url;

        if ($request->hasFile('image')) {
            // Delete old image if it exists
            if ($service->image_path) {
                Storage::disk('public')->delete($service->image_path);
            }
            $imagePath = $request->file('image')->store('services', 'public');
            $imageUrl = url($imagePath);
        }

        $service->update([
            'title' => $validated['title'],
            'description' => $validated['description'],
            'icon' => $validated['icon'] ?? null,
            'order' => $validated['order'],
            'image_path' => $imagePath,
            'image_url' => $imageUrl,
            'type' => $validated['type'],
            'price' => $validated['price'],
            'is_visible' => $validated['is_visible'] ?? true,
        ]);

        if (request()->wantsJson()) {
            return response()->json($service);
        }
        return redirect()->route('admin.services.index')->with('flash.success', 'Service updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Service $service)
    {
        // Delete associated image if it exists
        if ($service->image_path) {
            Storage::disk('public')->delete($service->image_path);
        }

        $service->delete();

        if (request()->wantsJson()) {
            return response()->json(null, 204);
        }
        return redirect()->route('admin.services.index')->with('flash.success', 'Service deleted successfully.');
    }
}