<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Storage;

class SettingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $setting = Setting::firstOrCreate([]);
        return redirect()->route('admin.settings.edit', ['setting' => $setting->id]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Not needed as there's only one setting, handled by index redirecting to edit
        return redirect()->route('admin.settings.edit', ['setting' => Setting::firstOrCreate([])->id]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Not needed as there's only one setting, handled by update
        return redirect()->route('admin.settings.edit', ['setting' => Setting::firstOrCreate([])->id]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Setting $setting)
    {
        // Not needed as there's only one setting, handled by index
        return response()->json($setting);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Setting $setting)
    {
        return Inertia::render('Admin/Settings/Edit', [
            'setting' => $setting,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Setting $setting)
    {
        $validatedData = $request->validate([
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'company_name' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:255',
            'address' => 'nullable|string|max:255',
            'facebook_link' => 'nullable|url|max:255',
            'twitter_link' => 'nullable|url|max:255',
            'linkedin_link' => 'nullable|url|max:255',
            'instagram_link' => 'nullable|url|max:255',
            'youtube_link' => 'nullable|url|max:255',
            'years_of_experience' => 'nullable|integer',
            'happy_clients' => 'nullable|integer',
            'projects_completion' => 'nullable|integer',
            'about_us_description' => 'nullable|string',
        ]);

        if ($request->hasFile('logo')) {
            if ($setting->logo_path) {
                Storage::disk('public')->delete($setting->logo_path);
            }
            $logoPath = $request->file('logo')->store('settings', 'public');
            $logoUrl = Storage::disk('public')->url($logoPath);
            $validatedData['logo_path'] = $logoPath;
            $validatedData['logo_url'] = $logoUrl;
        } else {
            // If no new logo is uploaded, retain the existing logo_path and logo_url
            $validatedData['logo_path'] = $setting->logo_path;
            $validatedData['logo_url'] = $setting->logo_url;
        }

        $setting->update($validatedData);

        if (request()->wantsJson()) {
            return response()->json($setting);
        }
        return redirect()->route('admin.settings.edit', ['setting' => $setting->id])->with('success', 'Settings updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Setting $setting)
    {
        // Not needed as there's only one setting
        return redirect()->route('admin.settings.index');
    }
}
