<?php

namespace App\Http\Controllers;

use App\Models\Team;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class TeamController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (request()->wantsJson()) {
            $teams = Team::orderBy('order')->get();
            return response()->json($teams);
        }
        return Inertia::render('Admin/Teams/Index', [
            'teams' => Team::orderBy('order')->get(),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/Teams/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'position' => 'required|string|max:255',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'description' => 'nullable|string',
            'facebook_link' => 'nullable|string|max:255',
            'twitter_link' => 'nullable|string|max:255',
            'linkedin_link' => 'nullable|string|max:255',
            'order' => 'required|integer',
        ]);

        $imagePath = $request->file('image')->store('teams', 'public');
        $imageUrl = Storage::disk('public')->url($imagePath);

        $team = Team::create([
            'name' => $request->name,
            'position' => $request->position,
            'image_path' => $imagePath,
            'image_url' => $imageUrl,
            'description' => $request->description,
            'facebook_link' => $request->facebook_link,
            'twitter_link' => $request->twitter_link,
            'linkedin_link' => $request->linkedin_link,
            'order' => $request->order,
        ]);

        if (request()->wantsJson()) {
            return response()->json($team, 201);
        }
        return redirect()->route('admin.teams.index')->with('success', 'Team member created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $team = Team::findOrFail($id);
        return response()->json($team);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Team $team)
    {
        return Inertia::render('Admin/Teams/Edit', [
            'team' => $team,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Team $team)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'position' => 'required|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'description' => 'nullable|string',
            'facebook_link' => 'nullable|string|max:255',
            'twitter_link' => 'nullable|string|max:255',
            'linkedin_link' => 'nullable|string|max:255',
            'order' => 'required|integer',
        ]);

        if ($request->hasFile('image')) {
            Storage::disk('public')->delete($team->image_path);
            $imagePath = $request->file('image')->store('teams', 'public');
            $imageUrl = Storage::disk('public')->url($imagePath);
            $team->image_path = $imagePath;
            $team->image_url = $imageUrl;
        }

        $team->update([
            'name' => $request->name,
            'position' => $request->position,
            'description' => $request->description,
            'facebook_link' => $request->facebook_link,
            'twitter_link' => $request->twitter_link,
            'linkedin_link' => $request->linkedin_link,
            'order' => $request->order,
        ]);

        if (request()->wantsJson()) {
            return response()->json($team);
        }
        return redirect()->route('admin.teams.index')->with('success', 'Team member updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Team $team)
    {
        Storage::disk('public')->delete($team->image_path);
        $team->delete();

        if (request()->wantsJson()) {
            return response()->json(null, 204);
        }
        return redirect()->route('admin.teams.index')->with('success', 'Team member deleted successfully.');
    }
}
