<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use App\Models\Setting;

class PaymentConfigServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // Ensure the application is not running in console (e.g., during migrations)
        // to avoid issues with database access before tables are ready.
        if ($this->app->runningInConsole()) {
            return;
        }

        try {
            \Log::info('Loading payment gateway settings...');
            $settings = Setting::first(); // Assuming a single settings record

            if ($settings && $settings->payment_gateways) {
                \Log::info('Payment gateways settings found.', ['settings' => $settings->payment_gateways]);
                $paymentGateways = $settings->payment_gateways;

                // Configure Paystack
                if (isset($paymentGateways['paystack'])) {
                    config()->set('paystack.publicKey', $paymentGateways['paystack']['public_key'] ?? null);
                    config()->set('paystack.secretKey', $paymentGateways['paystack']['secret_key'] ?? null);
                    config()->set('paystack.merchantEmail', $paymentGateways['paystack']['merchant_email'] ?? null);
                    config()->set('paystack.paymentUrl', $paymentGateways['paystack']['payment_url'] ?? 'https://api.paystack.co'); // Dynamically set paymentUrl
                }

                // Configure PayPal
                if (isset($paymentGateways['paypal'])) {
                    \Log::info('Configuring PayPal...');
                    $paypalConfig = config('paypal'); // Get existing PayPal config
                    $paypalDbConfig = $paymentGateways['paypal'];

                    // Update mode-specific credentials
                    if ($paypalDbConfig['mode'] === 'sandbox') {
                        $paypalConfig['mode'] = 'sandbox';
                        $paypalConfig['sandbox']['client_id'] = $paypalDbConfig['sandbox_client_id'] ?? null;
                        $paypalConfig['sandbox']['client_secret'] = $paypalDbConfig['sandbox_client_secret'] ?? null;
                        \Log::info('PayPal sandbox configured.', ['client_id' => $paypalDbConfig['sandbox_client_id'] ?? 'not set']);
                    } else {
                        $paypalConfig['mode'] = 'live';
                        $paypalConfig['live']['client_id'] = $paypalDbConfig['live_client_id'] ?? null;
                        $paypalConfig['live']['client_secret'] = $paypalDbConfig['live_client_secret'] ?? null;
                        \Log::info('PayPal live configured.', ['client_id' => $paypalDbConfig['live_client_id'] ?? 'not set']);
                    }
                    $paypalConfig['currency'] = $paypalDbConfig['currency'] ?? 'USD';

                    config()->set('paypal', $paypalConfig);
                } else {
                    \Log::warning('PayPal configuration not found in settings.');
                }
            } else {
                \Log::warning('No payment gateway settings found in the database.');
            }
        } catch (\Exception $e) {
            // Log the error if settings cannot be loaded from the database
            \Log::error('Failed to load payment gateway settings from database: ' . $e->getMessage());
        }
    }
}