<?php

namespace App\Services;

use Srmklive\PayPal\Services\PayPal as PayPalClient;

class PaypalService implements PaymentService
{
    protected $provider;

    public function __construct()
    {
        $this->provider = new PayPalClient();
        $this->provider->setApiCredentials(config('paypal'));
    }

    /**
     * Initialize a payment with PayPal.
     *
     * @param array $data
     * @return array
     */
    public function initializePayment(array $data): array
    {
        try {
            $this->provider->setAccessToken($this->provider->getAccessToken());
            $response = $this->provider->createOrder([
                "intent" => "CAPTURE",
                "application_context" => [
                    "return_url" => route('paypal.success'), // You'll need to define this route
                    "cancel_url" => route('paypal.cancel'), // You'll need to define this route
                ],
                "purchase_units" => [
                    [
                        "amount" => [
                            "currency_code" => config('paypal.currency'),
                            "value" => $data['total'],
                        ],
                        "custom_id" => $data['id'], // Store order_id here
                    ]
                ]
            ]);

            if (isset($response['id']) && $response['id'] !== null) {
                foreach ($response['links'] as $link) {
                    if ($link['rel'] === 'approve') {
                        return [
                            'status' => true,
                            'authorization_url' => $link['href'],
                            'reference' => $response['id'], // PayPal Order ID as reference
                        ];
                    }
                }
            }

            return ['status' => false, 'message' => $response['message'] ?? 'PayPal initialization failed.'];
        } catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Verify a payment with PayPal.
     *
     * @param string $reference (PayPal Order ID)
     * @return array
     */
    public function verifyPayment(string $reference): array
    {
        try {
            $response = $this->provider->capturePaymentOrder($reference);

            if (isset($response['status']) && $response['status'] === 'COMPLETED') {
                return [
                    'status' => true,
                    'data' => $response,
                ];
            }

            return ['status' => false, 'message' => $response['message'] ?? 'PayPal verification failed.'];
        } catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage()];
        }
    }
}