import React, { useEffect, useRef, useState } from 'react';

// Placeholder image component (using the provided URL)
const imagePlaceholder = 'https://www.promisebalm.com/assets/images/sliders/1698862701.png';

// --- Icon Definitions (Inline SVGs to replace react-icons/fa) ---
// 1. ChevronRight Icon
const ChevronRightIcon = (props) => (
    <svg {...props} xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
        <path d="m9 18 6-6-6-6"/>
    </svg>
);
// 2. HandHoldingMedical Icon (Simulated)
const HandHoldingMedicalIcon = (props) => (
    <svg {...props} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
        <path d="M2 12s3-1 6-2 6-2 10 0 4 2 4 2m-2-4v8m-14-6l-1 1-1 2 4 4 1-1m-4 1c0 2 2 4 4 4s4-2 4-4-2-4-4-4-4 2-4 4z"/>
    </svg>
);
// 3. Award Icon
const AwardIcon = (props) => (
    <svg {...props} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
        <circle cx="12" cy="8" r="7"/>
        <path d="M8.21 13.89a2 2 0 0 0 1.25 2.12l.66.24a2 2 0 0 1 1.26 2.12l.24.66a2 2 0 0 0 2.12 1.25l.66.24a2 2 0 0 1 2.12 1.26l.24.66a2 2 0 0 0 1.25 2.12l.66.24a2 2 0 0 1 2.12 1.26"/>
    </svg>
);
// 4. Users Icon
const UsersIcon = (props) => (
    <svg {...props} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
        <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/><circle cx="9" cy="7" r="4"/><path d="M22 21v-2a4 4 0 0 0-3-3.87m-3-1.63a4 4 0 0 1-1 4.5"/>
    </svg>
);
// 5. GraduationCap Icon
const GraduationCapIcon = (props) => (
    <svg {...props} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
        <path d="M21.43 14.83l-7.14-3.57a1.5 1.5 0 0 0-1.28 0l-7.14 3.57M21 16v3a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-3"/><path d="M3 14.28V18m18-3.72V18"/>
        <path d="M12 2a6 6 0 0 0-6 6v3.5l6 3 6-3V8a6 6 0 0 0-6-6z"/>
    </svg>
);


// --- Utility: Simulated useInView Hook (for smooth entry animations) ---
const useInView = (options) => {
    const ref = useRef(null);
    const [inView, setInView] = useState(false);

    useEffect(() => {
        const observer = new IntersectionObserver(([entry]) => {
            if (entry.isIntersecting) {
                setInView(true);
                // Stop observing once visible to maintain the animation state
                observer.unobserve(entry.target);
            }
        }, options);

        if (ref.current) {
            observer.observe(ref.current);
        }

        return () => {
            if (ref.current) {
                observer.unobserve(ref.current);
            }
        };
    }, [options]);

    return [ref, inView];
};

// --- Component for animating individual elements ---
const AnimatedDiv = ({ children, delay = 0, duration = 800, className }) => {
    const [ref, inView] = useInView({ threshold: 0.1 });

    // Tailwind classes for the animated state
    const transitionClass = `transition-all duration-[${duration}ms] ease-out`;
    // Using a subtle Y-translation for Framer Motion's lift-up effect
    const transformClass = inView 
        ? 'opacity-100 translate-y-0' 
        : 'opacity-0 translate-y-12';

    const style = {
        transitionDelay: `${delay}ms`,
    };

    return (
        <div ref={ref} className={`${className} ${transitionClass} ${transformClass}`} style={style}>
            {children}
        </div>
    );
};


const AboutPromisePalm = () => {

    // Ref for the main section animation
    const [sectionRef, sectionInView] = useInView({ threshold: 0.2 });

    // Data for the Value Proposition Cards
    const valueProps = [
        { 
            icon: HandHoldingMedicalIcon, // Replaced FaHandHoldingMedical
            title: "Premium Healthcare", 
            description: "We prioritize premium value on timely, high-quality care.",
            delay: 100 
        },
        { 
            icon: AwardIcon, // Replaced FaAward
            title: "Industry Standard", 
            description: "Setting a new standard in holistic health and wellness.",
            delay: 200 
        },
    ];

    return (
        <section 
            ref={sectionRef}
            className="relative py-20 md:py-32 bg-white overflow-hidden"
        >
            {/* Background Accent Element (Subtle, large curved shape, animated slide-in) */}
            <div className={`absolute top-0 right-0 h-4/5 w-full md:w-1/2 bg-green-50/50 rounded-bl-[10rem] transition-all duration-1000 ease-out ${
                sectionInView ? 'translate-x-0' : 'translate-x-full'
            }`} style={{ transform: sectionInView ? 'translateX(0)' : 'translateX(100%)' }}></div>
            
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
                
                {/* Section Title */}
                <AnimatedDiv delay={0} className="text-center mb-16">
                    <span className="inline-block px-4 py-1.5 mb-4 text-sm font-semibold text-green-700 border border-green-200 bg-green-50 rounded-full shadow-md">Who We Are</span>
                    <h2 className="text-4xl md:text-5xl font-extrabold text-gray-900 leading-tight mb-4">
                        About <span className="text-transparent bg-clip-text bg-gradient-to-r from-green-700 to-green-800">PromisePalm</span>
                    </h2>
                    <div className="w-24 h-1 bg-gradient-to-r from-green-700 to-green-800 mx-auto rounded-full"></div>
                </AnimatedDiv>
                
                <div className="lg:grid lg:grid-cols-12 lg:gap-16 items-start">
                    
                    {/* === Left Column: Text Content (Col-span 7) === */}
                    <div className="lg:col-span-7 text-gray-700 space-y-10 lg:pr-8 mb-12 lg:mb-0">
                        
                        {/* Core Description - Animated Slide-In */}
                        <AnimatedDiv delay={100} duration={900}>
                            <p className="text-xl font-light leading-relaxed border-l-4 border-green-500 pl-4">
                                We've got what you need! Promise Initiative Ghana and Sound Health (<span className="font-bold text-green-700">PROMISEBALM</span>) provide a combination of health, high-quality and personalized holistic medical care and wellness services virtually, in recognized health facilities, and in the environment of your choice.
                            </p>
                        </AnimatedDiv>
                        
                        {/* Value Propositions Grid */}
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-6">
                            {valueProps.map((prop, index) => (
                                <AnimatedDiv key={prop.title} delay={prop.delay + 200} duration={800}>
                                    <div className="flex items-start p-5 bg-white rounded-xl shadow-lg border-t-4 border-green-500 hover:shadow-xl transition-shadow duration-300">
                                        <div className="flex-shrink-0 w-12 h-12 bg-gradient-to-br from-green-500 to-green-600 rounded-full flex items-center justify-center shadow-md">
                                            {/* Render the icon component */}
                                            <prop.icon className="text-white text-xl w-6 h-6" /> 
                                        </div>
                                        <div className="ml-4">
                                            <h3 className="font-extrabold text-lg text-gray-900 mb-1">{prop.title}</h3>
                                            <p className="text-gray-600 text-sm">{prop.description}</p>
                                        </div>
                                    </div>
                                </AnimatedDiv>
                            ))}
                        </div>

                        {/* Stats Section - Cleaned up and Animated */}
                        <AnimatedDiv delay={700} duration={900} className="grid grid-cols-3 divide-x divide-gray-200 py-6 border-y border-gray-100 rounded-lg bg-gray-50/70 shadow-inner">
                            <div className="text-center px-4">
                                <div className="text-4xl font-extrabold text-transparent bg-clip-text bg-gradient-to-r from-green-600 to-green-700">500+</div>
                                <div className="text-sm text-gray-600 mt-1 font-medium">Happy Clients</div>
                            </div>
                            <div className="text-center px-4">
                                <div className="text-4xl font-extrabold text-transparent bg-clip-text bg-gradient-to-r from-green-600 to-green-700">50+</div>
                                <div className="text-sm text-gray-600 mt-1 font-medium">Expert Staff</div>
                            </div>
                            <div className="text-center px-4">
                                <div className="text-4xl font-extrabold text-transparent bg-clip-text bg-gradient-to-r from-green-600 to-green-700">10+</div>
                                <div className="text-sm text-gray-600 mt-1 font-medium">Years Experience</div>
                            </div>
                        </AnimatedDiv>
                    </div>
                    
                    {/* === Right Column: Visual & School Promotion (Col-span 5) === */}
                    <div className="lg:col-span-5 relative lg:pl-4">
                        <AnimatedDiv delay={400} duration={1000}>
                            <div className="relative rounded-3xl overflow-hidden shadow-2xl transition-all duration-500 hover:shadow-green-500/50">
                                {/* Image with slightly smaller height for better visual balance */}
                                <img
                                    src={imagePlaceholder}
                                    alt="Students in a healthcare class, focused on their studies. PB School logo overlay."
                                    className="w-full h-[450px] object-cover transition-transform duration-700 hover:scale-[1.02]"
                                />
                                
                                {/* Gradient Overlay for depth and text contrast */}
                                <div className="absolute inset-0 bg-gradient-to-t from-gray-900/80 via-transparent to-transparent"></div>
                                
                                {/* School Promotion Card */}
                                <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 text-center text-white p-6 backdrop-filter backdrop-blur-sm bg-black/10 rounded-xl">
                                    {/* School Icon */}
                                    <div className="w-20 h-20 mx-auto mb-4 bg-green-600 p-2 rounded-full flex items-center justify-center shadow-xl">
                                        <GraduationCapIcon className="text-white text-3xl w-8 h-8" strokeWidth="1.5" />
                                    </div>

                                    {/* Branding Text */}
                                    <h3 className="text-6xl sm:text-7xl font-black tracking-tight leading-none">
                                        <span className="text-green-400">PB</span> <span className="text-gray-50">SCHOOL</span>
                                    </h3>
                                    <p className="mt-2 text-sm font-light text-green-200">Excellence in Health Education</p>

                                    {/* CTA Link for the School */}
                                    <a
                                        href="#pb-school"
                                        className="mt-5 inline-block text-white font-bold border-b-2 border-green-400 hover:text-green-400 transition duration-300 text-md group/school"
                                        aria-label="Visit PB School page"
                                    >
                                        <span className='group-hover/school:tracking-wide transition-all duration-300'>Start Your Career Journey</span> &rarr;
                                    </a>
                                </div>
                            </div>
                        </AnimatedDiv>
                        
                        {/* Abstract Floating Element (for visual interest) */}
                        <div className="absolute -bottom-8 -right-8 w-24 h-24 bg-green-200 rounded-full opacity-60 blur-2xl animate-pulse-slow"></div>
                    </div>
                </div>
            </div>
        </section>
    );
};

export default AboutPromisePalm;