import { useState } from 'react';
import { motion } from 'framer-motion';
import { FaMapMarkerAlt, FaPhone, FaEnvelope, FaClock, FaPaperPlane } from 'react-icons/fa';

// Animation variants
const containerVariants = {
    hidden: { opacity: 0 },
    visible: {
        opacity: 1,
        transition: {
            staggerChildren: 0.2,
            delayChildren: 0.3
        }
    }
};

const itemVariants = {
    hidden: { y: 20, opacity: 0 },
    visible: {
        y: 0,
        opacity: 1,
        transition: { duration: 0.5, ease: "easeOut" }
    }
};

const ContactSection = ({ setting }) => {
    const [formData, setFormData] = useState({
        name: '',
        email: '',
        message: ''
    });

    const handleChange = (e) => {
        const { name, value } = e.target;
        setFormData(prev => ({ ...prev, [name]: value }));
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        // Form submission logic would go here
        console.log('Form submitted:', formData);
        // Reset form
        setFormData({ name: '', email: '', message: '' });
        // Show success message
        alert('Thank you for your message! We will get back to you soon.');
    };

    // Contact information
    const contactInfo = [
        {
            icon: <FaMapMarkerAlt className="text-xl" />,
            title: "Visit Us",
            details: setting ? setting?.address : "123 Main St, Anytown, USA"
        },
        {
            icon: <FaPhone className="text-xl" />,
            title: "Call Us",
            details: setting ? setting?.phone : "+1 (555) 123-4567"
        },
        {
            icon: <FaEnvelope className="text-xl" />,
            title: "Email Us",
            details: setting ? setting?.email : "hello@promisebalm.com"
        },
        {
            icon: <FaClock className="text-xl" />,
            title: "Business Hours",
            details: "Monday - Friday: 9am - 6pm"
        }
    ];

    return (
        <section className="py-24 bg-gradient-to-b from-green-50 to-white relative overflow-hidden" id="contact">
            {/* Background pattern */}
            <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIxMDAlIiBoZWlnaHQ9IjEwMCUiPjxkZWZzPjxwYXR0ZXJuIGlkPSJwYXR0ZXJuIiB3aWR0aD0iNDAiIGhlaWdodD0iNDAiIHBhdHRlcm5Vbml0cz0idXNlclNwYWNlT25Vc2UiIHBhdHRlcm5UcmFuc2Zvcm09InJvdGF0ZSg0NSkiPjxyZWN0IHg9IjAiIHk9IjAiIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0icmdiYSgxNiwgMTg1LDEyOSwwLjA1KSIvPjwvcGF0dGVybj48L2RlZnM+PHJlY3Qgd2lkdGg9IjEwMCUiIGhlaWdodD0iMTAwJSIgZmlsbD0idXJsKCNwYXR0ZXJuKSIvPjwvc3ZnPg==')] opacity-30"></div>
            
            {/* Decorative elements */}
            <div className="absolute top-20 left-10 w-24 h-24 rounded-full bg-gradient-to-r from-green-300 to-emerald-300 opacity-20 blur-xl"></div>
            <div className="absolute bottom-20 right-10 w-32 h-32 rounded-full bg-gradient-to-r from-emerald-300 to-green-300 opacity-20 blur-xl"></div>
            
            <div className="container mx-auto px-6 relative z-10">
                {/* Section header */}
                <motion.div 
                    className="text-center mb-16"
                    initial={{ opacity: 0, y: -20 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    transition={{ duration: 0.7 }}
                    viewport={{ once: true }}
                >
                    {/* Ornament above title */}
                    <div className="flex justify-center mb-6">
                        <div className="w-24 h-1 bg-gradient-to-r from-green-500 to-emerald-500 rounded-full"></div>
                        <div className="w-4 h-4 mx-2 rounded-full bg-gradient-to-r from-green-500 to-emerald-500"></div>
                        <div className="w-24 h-1 bg-gradient-to-r from-green-500 to-emerald-500 rounded-full"></div>
                    </div>
                    
                    <h2 className="text-5xl font-bold text-gray-800 mb-4 relative inline-block">
                        <span className="relative z-10">Get In Touch</span>
                        <span className="absolute bottom-0 left-0 w-full h-3 bg-gradient-to-r from-green-300/30 to-emerald-300/30 rounded-full -z-10"></span>
                    </h2>
                    
                    {/* Divider */}
                    <div className="flex justify-center mt-6">
                        <div className="h-1 w-32 bg-gradient-to-r from-green-500 to-emerald-500 rounded-full"></div>
                    </div>
                    
                    <p className="mt-8 text-xl text-gray-600 max-w-3xl mx-auto">
                        We'd love to hear from you. Send us a message and we'll respond as soon as possible.
                    </p>
                </motion.div>
                
                {/* Contact content */}
                <motion.div 
                    className="grid grid-cols-1 lg:grid-cols-2 gap-12"
                    variants={containerVariants}
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true, amount: 0.2 }}
                >
                    {/* Contact Form */}
                    <motion.div 
                        className="bg-white rounded-2xl shadow-lg p-8 border border-green-100/50"
                        variants={itemVariants}
                        whileHover={{ y: -5 }}
                    >
                        <div className="mb-8">
                            <h3 className="text-2xl font-bold text-gray-800 mb-2">Send Us a Message</h3>
                            <p className="text-gray-600">Fill out the form below and we'll get back to you as soon as possible.</p>
                        </div>
                        
                        <form onSubmit={handleSubmit}>
                            <div className="mb-6">
                                <label htmlFor="name" className="block text-gray-700 font-medium mb-2">Your Name</label>
                                <motion.input
                                    type="text"
                                    id="name"
                                    name="name"
                                    value={formData.name}
                                    onChange={handleChange}
                                    placeholder="John Doe"
                                    className="w-full p-4 border border-green-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-400 focus:border-transparent transition duration-300"
                                    required
                                    whileFocus={{ scale: 1.02 }}
                                />
                            </div>
                            
                            <div className="mb-6">
                                <label htmlFor="email" className="block text-gray-700 font-medium mb-2">Your Email</label>
                                <motion.input
                                    type="email"
                                    id="email"
                                    name="email"
                                    value={formData.email}
                                    onChange={handleChange}
                                    placeholder="john@example.com"
                                    className="w-full p-4 border border-green-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-400 focus:border-transparent transition duration-300"
                                    required
                                    whileFocus={{ scale: 1.02 }}
                                />
                            </div>
                            
                            <div className="mb-8">
                                <label htmlFor="message" className="block text-gray-700 font-medium mb-2">Your Message</label>
                                <motion.textarea
                                    id="message"
                                    name="message"
                                    value={formData.message}
                                    onChange={handleChange}
                                    placeholder="Tell us more about your project..."
                                    rows="5"
                                    className="w-full p-4 border border-green-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-400 focus:border-transparent transition duration-300"
                                    required
                                    whileFocus={{ scale: 1.02 }}
                                ></motion.textarea>
                            </div>
                            
                            <motion.button
                                type="submit"
                                className="w-full py-4 px-6 bg-gradient-to-r from-green-500 to-emerald-500 text-white font-bold rounded-lg shadow-md hover:shadow-lg transition-all duration-300 flex items-center justify-center"
                                whileHover={{ scale: 1.03 }}
                                whileTap={{ scale: 0.98 }}
                            >
                                <span>Send Message</span>
                                <FaPaperPlane className="ml-2" />
                            </motion.button>
                        </form>
                    </motion.div>
                    
                    {/* Contact Card */}
                    <motion.div 
                        className="bg-gradient-to-br from-green-500 to-emerald-500 rounded-2xl shadow-xl p-8 text-white"
                        variants={itemVariants}
                        whileHover={{ y: -5 }}
                    >
                        <div className="mb-8">
                            <h3 className="text-2xl font-bold mb-2">Contact Information</h3>
                            <p className="text-green-100">Feel free to reach out to us through any of the following channels.</p>
                        </div>
                        
                        <div className="space-y-6">
                            {contactInfo.map((info, index) => (
                                <motion.div 
                                    key={index}
                                    className="flex items-start"
                                    whileHover={{ x: 5 }}
                                >
                                    <div className="flex-shrink-0 w-12 h-12 rounded-full bg-white/20 flex items-center justify-center mr-4">
                                        {info.icon}
                                    </div>
                                    <div>
                                        <h4 className="font-bold text-lg">{info.title}</h4>
                                        <p className="text-green-100">{info.details}</p>
                                    </div>
                                </motion.div>
                            ))}
                        </div>
                        
                        {/* Map placeholder */}
                        <div className="mt-8 rounded-xl overflow-hidden h-48 bg-white/10">
                            <div className="w-full h-full flex items-center justify-center text-green-100">
                                <div className="text-center">
                                    <FaMapMarkerAlt className="text-3xl mx-auto mb-2" />
                                    <p>Interactive Map</p>
                                </div>
                            </div>
                        </div>
                        
                        {/* Social links */}
                        <div className="mt-8 pt-6 border-t border-white/20">
                            <p className="font-bold mb-4">Connect With Us</p>
                            <div className="flex space-x-4">
                                {['facebook', 'twitter', 'linkedin', 'instagram'].map((social, index) => (
                                    <motion.a
                                        key={index}
                                        href="#"
                                        className="w-10 h-10 rounded-full bg-white/20 flex items-center justify-center"
                                        whileHover={{ 
                                            backgroundColor: "rgba(255,255,255,0.3)",
                                            scale: 1.1,
                                            rotate: 5
                                        }}
                                        whileTap={{ scale: 0.9 }}
                                    >
                                        <div className="w-5 h-5 bg-white rounded-full"></div>
                                    </motion.a>
                                ))}
                            </div>
                        </div>
                    </motion.div>
                </motion.div>
            </div>
        </section>
    );
};

export default ContactSection;