import React from 'react';
import { motion } from 'framer-motion';
import { CalendarIcon, ClockIcon, MapPinIcon } from '@heroicons/react/24/outline'; // Clean Heroicons

const itemVariants = {
    hidden: { y: 15, opacity: 0 }, // Reduced y distance
    visible: {
        y: 0,
        opacity: 1,
        transition: { duration: 0.4, ease: "easeOut" },
    },
};

const EventCard = ({ event, itemVariants, onBookClick }) => {

    // --- New: Currency Format Helper (Centralized) ---
    const formatCurrency = (price) => {
        // Use Intl.NumberFormat for robust, localized currency formatting
        return new Intl.NumberFormat('en-GH', {
            style: 'currency',
            currency: 'GHS', // Ghana Cedi
            minimumFractionDigits: 2,
        }).format(price);
    };

    // Helper function to format the date part (e.g., 20 Oct)
    const formatDate = (dateString) => {
        const date = new Date(dateString);
        return date.toLocaleDateString(undefined, { day: '2-digit', month: 'short' });
    };

    // Helper function to format the time part (e.g., 10:00 AM)
    const formatTime = (dateString) => {
        const options = { hour: '2-digit', minute: '2-digit', hour12: true };
        return new Date(dateString).toLocaleTimeString(undefined, options);
    };

    // Image URL parser (kept from your original code)
    const imageUrl = (imageString) => {
        let imageUrls = [];
        try {
            const parsedArray = JSON.parse(imageString);
            if (Array.isArray(parsedArray)) {
                imageUrls = parsedArray;
            }
            return imageUrls;
        } catch (e) {
            console.error("Error parsing image JSON string:", e);
            return [];
        }
    };

    const firstImageUrl = imageUrl(event.images)[0];

    return (
        <motion.div
            // Premium look: slightly deeper shadow, clean structure
            className="bg-white rounded-xl overflow-hidden shadow-lg hover:shadow-xl transition-all duration-300 group flex flex-col h-full"
            variants={itemVariants}
            whileHover={{ y: -2 }} // Subtle lift on hover
            whileTap={{ scale: 0.99 }}
        >
            {/* Event Image & Prominent Date/Price Block */}
            <div className="relative h-48 bg-gray-100">
                <img
                    src={firstImageUrl}
                    alt={event.title}
                    className="w-full h-full object-cover transition-transform duration-300 group-hover:scale-105"
                    onError={(e) => {
                        e.target.onerror = null;
                        e.target.src = `https://placehold.co/600x400/374151/F9FAFB?text=${encodeURIComponent(event.title)}`; 
                    }}
                />

                {/* Date Tag - Modern, prominent display on the image */}
                <div className="absolute top-0 right-0 p-4">
                    <div className="bg-white shadow-xl rounded-lg p-2 text-center transform group-hover:scale-105 transition-transform duration-300">
                        <span className="text-2xl font-extrabold text-emerald-600 leading-none block">
                            {new Date(event.date).getDate()}
                        </span>
                        <span className="text-xs font-semibold uppercase text-slate-500 leading-none block">
                            {new Date(event.date).toLocaleDateString(undefined, { month: 'short' })}
                        </span>
                    </div>
                </div>

                {/* Price Badge - Bottom Left */}
                <div className="absolute bottom-0 left-0 p-4">
                    {event.type === "free" ? (
                        <span className="bg-emerald-600 text-white text-sm font-bold px-3 py-1.5 rounded-full shadow-lg">
                            FREE
                        </span>
                    ) : (
                        // Displaying Ghana Cedi symbol using the new helper
                        <span className="bg-green-700 text-white text-sm font-bold px-3 py-1.5 rounded-full shadow-lg">
                            {formatCurrency(event.price)}
                        </span>
                    )}
                </div>
            </div>


            {/* Event Details */}
            <div className="p-6 md:p-6 flex-grow flex flex-col">

                <h3 className="text-xl md:text-2xl font-extrabold text-slate-900 mb-2 leading-snug">
                    {event.title}
                </h3>
                
                {/* Key Info Section: Using clean Heroicons */}
                <div className="text-base text-slate-700 space-y-2 mb-4 pt-3 border-t border-gray-100">
                    
                    <div className="flex items-center">
                        <ClockIcon className="w-5 h-5 mr-3 text-emerald-600 flex-shrink-0" />
                        <span className="font-medium text-slate-700">
                            {formatTime(event.date)}
                        </span>
                    </div>

                    <div className="flex items-center">
                        <MapPinIcon className="w-5 h-5 mr-3 text-emerald-600 flex-shrink-0" />
                        <span className="truncate text-slate-600">{event.location}</span>
                    </div>
                </div>
                
                <p className="text-slate-600 text-sm mb-6 line-clamp-3 flex-grow">{event.description}</p>
                
                {/* Button: Stays primary color */}
                <motion.button
                    className="w-full bg-emerald-600 hover:bg-emerald-700 text-white font-semibold py-3 px-4 rounded-xl transition-colors duration-200 shadow-md hover:shadow-lg mt-auto"
                    whileHover={{ scale: 1.01 }}
                    whileTap={{ scale: 0.98 }}
                    onClick={() => onBookClick(event)}
                >
                    Register Now
                </motion.button>
            </div>
        </motion.div>
    );
};

export default EventCard;