import { useState } from 'react';
import { motion } from 'framer-motion';
import { FaImages, FaTimes, FaSearchPlus } from 'react-icons/fa';

const GalleryItems = ({ galleryImages }) => {
    const [selectedImage, setSelectedImage] = useState(null);
    const containerVariants = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: { staggerChildren: 0.1, delayChildren: 0.2 }
        }
    };

    const itemVariants = {
        hidden: { y: 30, opacity: 0 },
        visible: { y: 0, opacity: 1, transition: { duration: 0.6, ease: "easeOut" } }
    };
    
    return (
        <>
            <motion.div
                className="py-16 bg-gradient-to-br from-gray-50 to-gray-100"
                initial={{ opacity: 0 }}
                whileInView={{ opacity: 1 }}
                viewport={{ once: true }}
            >
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Section Header */}
                    <div className="text-center mb-16">
                        <div className="flex items-center justify-center mb-6">
                            <div className="w-16 h-1 bg-gradient-to-t from-yellow-700 to-red-700 rounded-full mr-4"></div>
                            <FaImages className="text-yellow-700 text-3xl" />
                            <div className="w-16 h-1 bg-gradient-to-t from-yellow-700 to-red-700 rounded-full ml-4"></div>
                        </div>
                        <h2 className="text-4xl font-bold text-gray-900 mb-4">Our Gallery</h2>
                        <p className="text-lg text-gray-600 max-w-2xl mx-auto">
                            Explore our collection of professional images showcasing our work, team, and achievements.
                        </p>
                    </div>
                    
                    {/* Gallery Grid */}
                    <motion.div
                        className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8"
                        variants={containerVariants}
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                    >
                        {galleryImages.map((image) => (
                            <motion.div
                                key={image.id}
                                className="bg-white rounded-xl overflow-hidden shadow-lg border border-gray-200 transition-all duration-500 hover:shadow-2xl hover:border-blue-200 group"
                                variants={itemVariants}
                                whileHover={{ y: -10 }}
                                onClick={() => setSelectedImage(image)}
                            >
                                <div className="relative overflow-hidden">
                                    {/* Image */}
                                    <img 
                                        src={image.url} 
                                        alt={image.title}
                                        className="w-full h-[400px] object-cover transition-transform duration-700 group-hover:scale-110"
                                        loading="lazy"
                                    />
                                    
                                    {/* Overlay */}
                                    <div className="absolute inset-0 bg-gradient-to-t from-gray-900/90 via-gray-900/40 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500 flex flex-col justify-end p-6">
                                        <div className="transform translate-y-4 group-hover:translate-y-0 transition-transform duration-500">
                                            <h3 className="font-bold text-xl text-white mb-1">{image.title}</h3>
                                            <p className="text-blue-300 text-sm font-medium">{image.category}</p>
                                        </div>
                                        
                                        {/* View Button */}
                                        <div className="mt-4 transform translate-y-8 group-hover:translate-y-0 transition-transform duration-500">
                                            <button className="flex items-center text-white bg-blue-600 hover:bg-blue-700 px-4 py-2 rounded-lg text-sm font-medium transition-colors duration-300">
                                                <FaSearchPlus className="mr-2" />
                                                View Details
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </motion.div>
                        ))}
                    </motion.div>
                    
                    {/* Empty State */}
                    {galleryImages.length === 0 && (
                        <div className="text-center py-16">
                            <div className="inline-flex items-center justify-center w-24 h-24 rounded-full bg-blue-50 mb-6">
                                <FaImages className="text-blue-500 text-4xl" />
                            </div>
                            <h3 className="text-2xl font-bold text-gray-900 mb-2">No Gallery Items</h3>
                            <p className="text-gray-600 max-w-md mx-auto">
                                There are currently no images in the gallery. Check back soon for updates.
                            </p>
                        </div>
                    )}
                </div>
            </motion.div>

            {/* Image Modal */}
            {selectedImage && (
                <div className="fixed inset-0 bg-black bg-opacity-95 z-50 flex items-center justify-center p-4">
                    <div className="relative max-w-6xl max-h-[90vh] w-full">
                        {/* Close Button */}
                        <button
                            onClick={() => setSelectedImage(null)}
                            className="absolute top-6 right-6 text-white bg-blue-600 hover:bg-blue-700 rounded-full p-3 transition-all duration-300 z-10 shadow-lg"
                            aria-label="Close image view"
                        >
                            <FaTimes className="text-xl" />
                        </button>
                        
                        {/* Modal Content */}
                        <div className="bg-white rounded-xl overflow-hidden shadow-2xl">
                            {/* Image Container */}
                            <div className="relative bg-gray-900">
                                <img 
                                    src={selectedImage.url} 
                                    alt={selectedImage.title}
                                    className="w-full h-auto max-h-[70vh] object-contain"
                                />
                                
                                {/* Image Info Overlay */}
                                <div className="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/90 to-transparent p-6">
                                    <h3 className="text-2xl font-bold text-white mb-2">{selectedImage.title}</h3>
                                    <p className="text-blue-300 font-medium">{selectedImage.category}</p>
                                </div>
                            </div>
                            
                            {/* Image Details */}
                            <div className="p-6 bg-gray-50">
                                <div className="flex justify-between items-center">
                                    <div>
                                        <h4 className="text-lg font-semibold text-gray-800">Image Details</h4>
                                    </div>
                                    <button 
                                        onClick={() => setSelectedImage(null)}
                                        className="px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg transition-colors duration-300"
                                    >
                                        Close
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            )}
        </>
    );
};

export default GalleryItems;