import React, { useState, useRef, useEffect } from 'react';
import { Link, usePage } from '@inertiajs/react';
import MiniCart from '@/Components/MiniCart'; // Make sure this path is correct

const Navbar = (props) => {
    const { auth, cartItems, cartCount } = usePage().props;
    // console.log('Navbar props:', usePage().props);

    const [isScrolled, setIsScrolled] = useState(false);
    const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
    const [searchOpen, setSearchOpen] = useState(false);
    const [profileDropdownOpen, setProfileDropdownOpen] = useState(false);
    const [isLoaded, setIsLoaded] = useState(false);
    const [isMiniCartOpen, setIsMiniCartOpen] = useState(false); // State for mini-cart

    const searchRef = useRef(null);
    const profileRef = useRef(null);
    const miniCartRef = useRef(null); // Ref for mini-cart to close on outside click

    // Check if the user is authenticated
    const isAuthenticated = auth && auth.user;


    // Handle scroll effect
    useEffect(() => {
        const handleScroll = () => {
            if (window.scrollY > 10) {
                setIsScrolled(true);
            } else {
                setIsScrolled(false);
            }
        };

        window.addEventListener('scroll', handleScroll);
        return () => window.removeEventListener('scroll', handleScroll);
    }, []);

    // Handle initial load animation
    useEffect(() => {
        setTimeout(() => setIsLoaded(true), 100);
    }, []);

    // Close search and profile dropdown when clicking outside
    useEffect(() => {
        const handleClickOutside = (event) => {
            // Close search
            if (searchRef.current && !searchRef.current.contains(event.target)) {
                setSearchOpen(false);
            }
            // Close profile dropdown
            if (profileRef.current && !profileRef.current.contains(event.target)) {
                setProfileDropdownOpen(false);
            }
        };

        document.addEventListener('mousedown', handleClickOutside);
        return () => document.removeEventListener('mousedown', handleClickOutside);
    }, []);

    const navLinks = [
        { name: 'Home', route: 'home', icon: (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"></path>
            </svg>
        )},
        { name: 'Events', route: 'events', icon: (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"></path>
            </svg>
        )},
        { name: 'School', route: 'school', icon: (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 14l9-5-9-5-9 5 9 5z"></path>
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 14l6.16-3.422a12.083 12.083 0 01.665 6.479A11.952 11.952 0 0012 20.055a11.952 11.952 0 00-6.824-2.998 12.078 12.078 0 01.665-6.479L12 14z"></path>
            </svg>
        )},
        { name: 'Services', route: 'services', icon: (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M21 13.255A23.931 23.931 0 0112 15c-3.183 0-6.22-.62-9-1.745M16 6V4a2 2 0 00-2-2h-4a2 2 0 00-2 2v2m4 6h.01M5 20h14a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"></path>
            </svg>
        )},
        { name: 'Shop', route: 'shop', icon: (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"></path>
            </svg>
        )},
        { name: 'Blog', route: 'blog.index', icon: (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"></path>
            </svg>
        )},
        { name: 'Contact', route: 'contact', icon: (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"></path>
            </svg>
        )},
    ];

    const userNavLinks = [
        { name: 'Dashboard', route: auth.user?.role === 500 ? 'admin.dashboard' : 'user.dashboard', icon: (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37a1.724 1.724 0 002.572-1.065z"></path><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path></svg>
        )},
        { name: 'My Bookings', route: 'user.bookings.index', icon: (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
            </svg>
        )},
        { name: 'My Tickets', route: 'user.tickets.index', icon: (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M15 5v2m0 4v2m0 4v2M12 5v2m0 4v2m0 4v2M9 5v2m0 4v2m0 4v2M19 4H5a2 2 0 00-2 2v14a2 2 0 002 2h14a2 2 0 002-2V6a2 2 0 00-2-2zm0 16H5V8h14v12z" />
            </svg>
        )},
        { name: 'My School Services', route: 'user.school-services.index', icon: (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 14l9-5-9-5-9 5 9 5z"></path>
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 14l6.16-3.422a12.083 12.083 0 01.665 6.479A11.952 11.952 0 0012 20.055a11.952 11.952 0 00-6.824-2.998 12.078 12.078 0 01.665-6.479L12 14z"></path>
            </svg>
        )},
    ];

    return (
        <>
            <nav className={`fixed top-0 left-0 right-0 z-50 transition-all duration-500 ${
                isScrolled 
                    ? 'bg-white/95 backdrop-blur-md shadow-xl' 
                    : 'bg-white shadow-lg'
            } ${isLoaded ? 'animate-fadeInDown' : 'opacity-0'}`}>
                {/* Animated Corporate Accent Border */}
                <div className="h-1 bg-gradient-to-r from-green-600 via-emerald-500 to-teal-600 relative overflow-hidden">
                    <div className="absolute inset-0 bg-gradient-to-r from-transparent via-white/30 to-transparent animate-pulse"></div>
                </div>

                {/* Secondary Accent Line */}
                <div className="h-px bg-gradient-to-r from-transparent via-gray-200 to-transparent"></div>

                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex justify-between h-16 md:h-20 lg:h-24">

                        {/* Logo Section with slideIn animation */}
                        <div className={`flex items-center ${isLoaded ? 'animate-slideInLeft' : 'opacity-0'}`}>
                            <div className="flex-shrink-0 flex items-center group">
                                <div className="flex items-center">
                                    {/* Logo Icon with bounce animation - smaller on mobile */}
                                    <div className="w-8 h-8 sm:w-10 sm:h-10 md:w-12 md:h-12 bg-gradient-to-br from-green-600 to-emerald-700 rounded-lg sm:rounded-xl flex items-center justify-center mr-2 sm:mr-3 shadow-lg group-hover:shadow-xl transform transition-all duration-300 group-hover:scale-110 hover-bounce">
                                        <svg className="h-4 w-4 sm:h-5 sm:w-5 md:h-7 md:w-7 text-white" fill="currentColor" viewBox="0 0 20 20">
                                            <path fillRule="evenodd" d="M5.05 4.05a7 7 0 119.9 9.9L10 18.9l-4.95-4.95a7 7 0 010-9.9zM10 11a2 2 0 100-4 2 2 0 000 4z" clipRule="evenodd" />
                                        </svg>
                                    </div>
                                    <Link href={route('home')} className="text-lg sm:text-xl md:text-2xl font-bold bg-gradient-to-r from-green-700 to-emerald-600 bg-clip-text text-transparent group-hover:from-green-600 group-hover:to-emerald-500 transition-all duration-300">
                                        PromisePalm
                                    </Link>
                                </div>
                            </div>
                        </div>

                        {/* Desktop Menu Items with flipIn animation */}
                        <div className={`hidden lg:flex lg:items-center lg:space-x-1 ${isLoaded ? 'animate-flipInX' : 'opacity-0'}`} style={{ animationDelay: '0.2s' }}>
                            {navLinks.map((link, index) => (
                                <Link
                                    key={link.route}
                                    href={route(link.route)}
                                    className={`group relative px-5 py-3 rounded-lg text-sm font-medium transition-all duration-300 flex items-center space-x-2 hover-rubberBand ${
                                        route().current(link.route)
                                            ? 'text-green-600'
                                            : 'text-gray-700 hover:text-green-600'
                                    }`}
                                    style={{ animationDelay: `${0.3 + index * 0.1}s` }}
                                >
                                    <span className={`transition-all duration-300 ${
                                        route().current(link.route) ? 'text-green-600 scale-110 animate-pulse-once' : 'text-gray-500 group-hover:text-green-600'
                                    }`}>
                                        {link.icon}
                                    </span>
                                    <span>{link.name}</span>
                                    
                                    {/* Unique Active State - Simple Underline with Dots */}
                                    {route().current(link.route) && (
                                        <div className="absolute -bottom-1 left-1/2 transform -translate-x-1/2 flex items-center space-x-1 animate-zoomIn">
                                            <div className="w-1.5 h-1.5 bg-green-600 rounded-full"></div>
                                            <div className="w-8 h-0.5 bg-gradient-to-r from-green-600 to-emerald-500 rounded-full"></div>
                                            <div className="w-1.5 h-1.5 bg-green-600 rounded-full"></div>
                                        </div>
                                    )}
                                    
                                    {/* Hover Underline */}
                                    {!route().current(link.route) && (
                                        <div className="absolute -bottom-1 left-1/2 transform -translate-x-1/2 w-0 h-0.5 bg-gradient-to-r from-green-600 to-emerald-500 rounded-full group-hover:w-8 transition-all duration-300"></div>
                                    )}
                                </Link>
                            ))}

                            <div className="flex items-center space-x-3 ml-6 pl-6 border-l border-gray-200">
                                {/* Conditional Render: Profile Dropdown for authenticated users */}
                                {isAuthenticated ? (
                                    <div className="relative" ref={profileRef}>
                                        <button
                                            onClick={() => setProfileDropdownOpen(!profileDropdownOpen)}
                                            className="p-2.5 rounded-full bg-green-500 hover:bg-green-600 text-white transition-all duration-300 shadow-md flex items-center justify-center hover-bounce"
                                        >
                                            {/* User Icon */}
                                            <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                                            </svg>
                                        </button>

                                        {/* Profile Dropdown Content */}
                                        <div className={`absolute right-0 mt-2 w-48 bg-white rounded-xl shadow-2xl border border-gray-100 overflow-hidden transition-all duration-300 ${
                                            profileDropdownOpen ? 'opacity-100 translate-y-0 visible animate-zoomIn' : 'opacity-0 -translate-y-2 invisible'
                                        }`}>
                                            <div className="py-2">
                                                <div className="px-4 py-2 text-sm text-gray-700 font-semibold truncate border-b border-gray-100">
                                                    Hello, {auth.user.name.split(' ')[0]}!
                                                </div>
                                                {userNavLinks.map((link) => (
                                                    <Link
                                                        key={link.route}
                                                        href={route(link.route)}
                                                        onClick={() => setProfileDropdownOpen(false)}
                                                        className="flex items-center px-4 py-2 text-sm text-gray-700 hover:bg-green-50 hover:text-green-600 transition-colors duration-200"
                                                    >
                                                        {link.icon}
                                                        <span className="ml-2">{link.name}</span>
                                                    </Link>
                                                ))}
                                                <Link
                                                    href={route('logout')}
                                                    method="post"
                                                    as="button"
                                                    onClick={() => setProfileDropdownOpen(false)}
                                                    className="w-full text-left flex items-center px-4 py-2 text-sm text-red-600 hover:bg-red-50 transition-colors duration-200"
                                                >
                                                    <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3v-4m0 0V7a3 3 0 013-3h4m4 0h6m-6 0v1"></path></svg>
                                                    Log Out
                                                </Link>
                                            </div>
                                        </div>
                                    </div>
                                ) : (
                                    /* Original Login/Register Buttons for guests */
                                    <>
                                        <Link
                                            href={route('login')}
                                            className={`group relative px-5 py-2.5 rounded-xl text-sm font-medium transition-all duration-300 flex items-center space-x-2 ${
                                                route().current('login')
                                                    ? 'text-green-600 border-b-2 border-green-600'
                                                    : 'text-gray-700 hover:text-green-600 hover:bg-green-50 hover:shadow-md hover:-translate-y-0.5'
                                            }`}
                                        >
                                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M11 16l-4-4m0 0l4-4m-4 4h14m-5 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h7a3 3 0 013 3v1"></path>
                                            </svg>
                                            <span>Login</span>
                                        </Link>
                                        <Link
                                            href={route('register')}
                                            className={`group relative px-5 py-2.5 rounded-xl text-sm font-medium transition-all duration-300 shadow-md flex items-center space-x-2 overflow-hidden ${
                                                route().current('register')
                                                    ? 'bg-gradient-to-r from-green-700 to-emerald-600 text-white shadow-lg'
                                                    : 'bg-gradient-to-r from-green-600 to-emerald-600 hover:from-green-700 hover:to-emerald-600 text-white hover:shadow-lg hover:scale-105'
                                            }`}
                                        >
                                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M18 9v3m0 0v3m0-3h3m-3 0h-3m-2-5a4 4 0 11-8 0 4 4 0 018 0zM3 20a6 6 0 0112 0v1H3v-1z"></path>
                                            </svg>
                                            <span>Register</span>
                                            {/* Shine effect */}
                                            <div className="absolute inset-0 rounded-xl bg-gradient-to-r from-transparent via-white/30 to-transparent transform -skew-x-12 -translate-x-full group-hover:translate-x-full transition-transform duration-700"></div>
                                        </Link>
                                    </>
                                )}
                            </div>
                        </div>

                        {/* Icons Section with slideIn animation */}
                        <div className={`flex items-center space-x-1 sm:space-x-2 ${isLoaded ? 'animate-slideInRight' : 'opacity-0'}`} style={{ animationDelay: '0.4s' }}>
                            {/* Search with dropdown - adjusted for mobile */}
                            
                            

                            {/* Shopping Cart Icon with Badge - adjusted for mobile */}
                            <div className="relative group" ref={miniCartRef}>
                                <button 
                                    onClick={() => setIsMiniCartOpen(!isMiniCartOpen)}
                                    className="p-2 sm:p-2.5 rounded-full text-gray-500 hover:text-green-600 hover:bg-green-50 transition-all duration-300 hover-bounce"
                                >
                                    {/* Cart Icon */}
                                    <svg className="h-5 w-5 sm:h-6 sm:w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z" />
                                    </svg>
                                </button>
                                {/* Cart Badge with animation - adjusted for mobile */}
                                <span className="absolute top-0 right-0 inline-flex items-center justify-center px-1.5 py-0.5 sm:px-2 sm:py-1 text-xs font-bold leading-none text-white transform translate-x-1/2 -translate-y-1/2 bg-gradient-to-r from-green-600 to-emerald-600 rounded-full shadow-md group-hover:scale-110 group-hover:animate-bounce-once transition-all duration-300">
                                    {cartCount}
                                </span>
                            </div>

                            {/* Mobile menu button - adjusted for mobile */}
                            <button 
                                onClick={() => {
                                    setMobileMenuOpen(!mobileMenuOpen);
                                    setSearchOpen(false); // Close search when opening mobile menu
                                    setProfileDropdownOpen(false); // Close profile when opening mobile menu
                                }}
                                className="lg:hidden p-2 sm:p-2.5 rounded-full text-gray-500 hover:text-green-600 hover:bg-green-50 transition-all duration-300 hover-rubberBand"
                            >
                                <svg className="h-5 w-5 sm:h-6 sm:w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                    {mobileMenuOpen ? (
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M6 18L18 6M6 6l12 12" />
                                    ) : (
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M4 6h16M4 12h16M4 18h16" />
                                    )}
                                </svg>
                            </button>
                            
                            {/* NEW: Mobile Profile Icon for authenticated users (only shown on mobile/tablet) */}
                             {isAuthenticated && (
                                <div className="lg:hidden relative">
                                    <button
                                        onClick={() => {
                                            setProfileDropdownOpen(!profileDropdownOpen);
                                            setMobileMenuOpen(false); // Close mobile menu if open
                                        }}
                                        className="p-2 sm:p-2.5 rounded-full bg-green-500 hover:bg-green-600 text-white transition-all duration-300 shadow-md flex items-center justify-center hover-bounce"
                                    >
                                        <svg className="h-5 w-5 sm:h-6 sm:w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                                        </svg>
                                    </button>
                                </div>
                            )}
                        </div>
                    </div>
                </div>

                {/* Bottom Corporate Border */}
                <div className="border-b border-gray-100"></div>

                {/* Decorative Bottom Pattern */}
                <div className="h-px bg-gradient-to-r from-transparent via-green-100 to-transparent opacity-50"></div>

                {/* Mobile menu with slideIn animation - improved for mobile */}
                <div className={`lg:hidden overflow-hidden transition-all duration-500 ${
                    mobileMenuOpen ? 'max-h-screen animate-slideInLeft' : 'max-h-0'
                }`}>
                    <div className="px-2 sm:px-4 pt-2 pb-6 space-y-1 sm:space-y-2 bg-gradient-to-b from-gray-50 to-white">
                        {navLinks.map((link, index) => (
                            <Link
                                key={link.route}
                                href={route(link.route)}
                                className={`group flex items-center space-x-3 px-3 sm:px-4 py-3 rounded-xl text-sm sm:text-base font-medium transition-all duration-300 ${
                                    route().current(link.route)
                                        ? 'text-green-600 bg-green-50 border-l-4 border-green-600 shadow-sm animate-pulse-once'
                                        : 'text-gray-700 hover:text-green-600 hover:bg-green-50'
                                }`}
                                style={{ animationDelay: `${index * 0.05}s` }}
                                onClick={() => setMobileMenuOpen(false)}
                            >
                                <span className={`transition-all duration-300 ${
                                    route().current(link.route) ? 'text-green-600 scale-110' : 'text-gray-500 group-hover:text-green-600'
                                }`}>
                                    {link.icon}
                                </span>
                                <span>{link.name}</span>
                            </Link>
                        ))}
                        
                        {/* Conditional Render: Login/Register for guests in mobile menu */}
                        {!isAuthenticated && (
                            <div className="pt-4 mt-4 border-t border-gray-200 space-y-2 sm:space-y-3">
                                <Link
                                    href={route('login')}
                                    className={`group flex items-center space-x-3 px-3 sm:px-4 py-3 rounded-xl text-sm sm:text-base font-medium transition-all duration-300 hover-bounce ${
                                        route().current('login')
                                            ? 'text-green-600 bg-green-50 border-l-4 border-green-600 shadow-sm'
                                            : 'text-gray-700 hover:text-green-600 hover:bg-green-50'
                                    }`}
                                    onClick={() => setMobileMenuOpen(false)}
                                >
                                    <svg className="w-4 h-4 sm:w-5 sm:h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M11 16l-4-4m0 0l4-4m-4 4h14m-5 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h7a3 3 0 013 3v1"></path>
                                    </svg>
                                    <span>Login</span>
                                </Link>
                                <Link
                                    href={route('register')}
                                    className={`group flex items-center space-x-3 px-3 sm:px-4 py-3 rounded-xl text-sm sm:text-base font-medium transition-all duration-300 hover-rubberBand ${
                                        route().current('register')
                                            ? 'bg-gradient-to-r from-green-700 to-emerald-600 text-white shadow-lg animate-pulse-once'
                                            : 'bg-gradient-to-r from-green-600 to-emerald-600 hover:from-green-700 hover:to-emerald-600 text-white'
                                    }`}
                                    onClick={() => setMobileMenuOpen(false)}
                                >
                                    <svg className="w-4 h-4 sm:w-5 sm:h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M18 9v3m0 0v3m0-3h3m-3 0h-3m-2-5a4 4 0 11-8 0 4 4 0 018 0zM3 20a6 6 0 0112 0v1H3v-1z"></path>
                                    </svg>
                                    <span>Register</span>
                                </Link>
                            </div>
                        )}
                        
                        {/* NEW: Profile Dropdown Menu for authenticated users in mobile menu */}
                        {isAuthenticated && (
                            <div className="pt-4 mt-4 border-t border-gray-200 space-y-2 sm:space-y-3">
                                <Link
                                    href={route(auth.user.role === 500 ? 'admin.dashboard' : 'user.dashboard')}
                                    className="flex items-center space-x-3 px-3 sm:px-4 py-3 rounded-xl text-sm sm:text-base font-medium text-gray-700 hover:bg-green-50 hover:text-green-600 transition-colors duration-200"
                                    onClick={() => setMobileMenuOpen(false)}
                                >
                                     <svg className="w-4 h-4 sm:w-5 sm:h-5 text-gray-500 group-hover:text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37a1.724 1.724 0 002.572-1.065z"></path><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path></svg>
                                    <span>Dashboard</span>
                                </Link>
                                <Link
                                    href={route('logout')}
                                    method="post"
                                    as="button"
                                    className="w-full text-left flex items-center space-x-3 px-3 sm:px-4 py-3 rounded-xl text-sm sm:text-base font-medium text-red-600 hover:bg-red-50 transition-colors duration-200"
                                    onClick={() => setMobileMenuOpen(false)}
                                >
                                    <svg className="w-4 h-4 sm:w-5 sm:h-5 text-red-500" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3v-4m0 0V7a3 3 0 013-3h4m4 0h6m-6 0v1"></path></svg>
                                    <span>Log Out</span>
                                </Link>
                            </div>
                        )}
                    </div>
                </div>
            </nav>

            {/* Spacer to prevent content from being hidden behind the fixed navbar - adjusted for mobile */}
            <div className="h-16 md:h-20 lg:h-24"></div>

            <MiniCart cartItems={cartItems} isOpen={isMiniCartOpen} onClose={() => {
                console.log('Closing MiniCart');
                setIsMiniCartOpen(false);
            }} />
        </>
    );
};

export default Navbar;