import React from 'react';
import { motion } from 'framer-motion';

// --- Configuration ---
const containerVariants = {
    hidden: { opacity: 0 },
    visible: {
        opacity: 1,
        transition: {
            staggerChildren: 0.15,
            delayChildren: 0.3
        }
    }
};

const itemVariants = {
    hidden: { y: 20, opacity: 0 },
    visible: {
        y: 0,
        opacity: 1,
        transition: { duration: 0.5, ease: "easeOut" }
    }
};

// --- Sub-Component: PortfolioCard ---
const PortfolioCard = ({ item }) => {
    const vibrantGradient = "bg-gradient-to-br from-yellow-500 to-red-600";

    return (
        <motion.div 
            className="group relative overflow-hidden rounded-2xl shadow-xl transition-all duration-300 cursor-pointer"
            variants={itemVariants}
            whileHover={{ 
                y: -5, 
                rotate: 0.5,
                boxShadow: "0 20px 25px -5px rgba(251, 191, 36, 0.4), 0 10px 10px -5px rgba(251, 191, 36, 0.1)"
            }} 
            whileTap={{ scale: 0.98 }}
        >
            <div className="relative h-72 sm:h-80 overflow-hidden">
                <img 
                    src={item.image_url} // Changed from item.image
                    alt={item.title} 
                    className="w-full h-full object-cover transform group-hover:scale-105 transition-transform duration-500"
                    onError={(e) => { 
                        e.target.onerror = null; 
                        e.target.src = "https://placehold.co/600x400/99F6E4/042F2E?text=Project" 
                    }}
                />
                
                {/* Overlay without category */}
                <div className={`absolute inset-0 p-8 flex flex-col justify-end transition-opacity duration-500 ${vibrantGradient} opacity-0 group-hover:opacity-95`}>
                    <h3 className="text-2xl font-bold text-white mb-2">{item.title}</h3>
                    <p className="text-yellow-100 text-base mb-4 italic">{item.description}</p>
                </div>
            </div>
            
            {/* Visible content without category */}
            <div className="p-6 bg-white border-t border-gray-100">
                <h3 className="text-xl font-bold text-gray-900 mb-3">{item.title}</h3>
                <p className="text-gray-600 text-sm">{item.description}</p>
            </div>
        </motion.div>
    );
};

// --- Main Component: PortfolioSection ---
const PortfolioSection = ({ portfolios }) => {
    const warmBackground = "bg-gradient-to-br from-white to-yellow-50";
    const vibrantGradient = "bg-gradient-to-r from-yellow-500 to-red-600";

    return (
        <section className={`py-20 lg:py-32 ${warmBackground}`} id="portfolio">
            <div className="container mx-auto px-6">
                <motion.div 
                    className="text-center mb-16 lg:mb-24"
                    initial={{ opacity: 0, y: -30 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    transition={{ duration: 0.7 }}
                    viewport={{ once: true }}
                >
                    <p className="text-lg font-semibold text-red-600 mb-3">
                        Our Hand-Picked Creations
                    </p>
                    <h2 className="text-5xl lg:text-6xl font-extrabold text-gray-800 mb-6">
                        Exquisite Portfolio
                    </h2>
                    <div className="flex justify-center mb-6">
                        <div className={`h-2 w-20 ${vibrantGradient} rounded-full`}></div>
                    </div>
                    <p className="mt-8 text-xl text-gray-600 max-w-4xl mx-auto leading-relaxed">
                        A showcase of our passion projects, where thoughtful design meets powerful execution.
                    </p>
                </motion.div>
                
                {/* Portfolio items grid */}
                <motion.div 
                    className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8"
                    variants={containerVariants}
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true, amount: 0.1 }}
                >
                    {portfolios && portfolios.length > 0 ? (
                        portfolios.map((item) => (
                            <PortfolioCard key={item.id} item={item} />
                        ))
                    ) : (
                        <div className="col-span-full text-center py-10">
                            <p className="text-xl text-gray-600">No portfolios available at the moment.</p>
                        </div>
                    )}
                </motion.div>
                
                {/* View All Projects Button */}
                {/* <div className="text-center mt-16">
                    <motion.button 
                        className={`px-10 py-4 text-lg ${vibrantGradient} text-white font-bold rounded-full shadow-xl shadow-red-300/70 transform transition-all duration-300 focus:outline-none focus:ring-4 focus:ring-red-500/50`}
                        whileHover={{ scale: 1.05, boxShadow: "0 15px 20px -3px rgba(239, 68, 68, 0.7)" }}
                        whileTap={{ scale: 0.95 }}
                    >
                        Explore More Creations
                    </motion.button>
                </div> */}
            </div>
        </section>
    );
};

export default PortfolioSection;