import React, { useState } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { FaShoppingCart, FaStar, FaTimes, FaExpand, FaChevronLeft, FaChevronRight, FaSearchPlus } from 'react-icons/fa';
import { useForm } from '@inertiajs/react';
import AddToCartModal from '@/Components/AddToCartModal';
import ProductCard from '@/Components/ProductCard';

// Animation variants
const containerVariants = {
    hidden: { opacity: 0 },
    visible: {
        opacity: 1,
        transition: {
            staggerChildren: 0.2,
            delayChildren: 0.3
        }
    }
};

// Main Products Section Component
const ProductsSection = ({ products }) => {
    const { post } = useForm();
    const [selectedProduct, setSelectedProduct] = useState(null);
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [showToast, setShowToast] = useState(false);

    

    const handleProductClick = (product) => {
        setSelectedProduct(product);
        setIsModalOpen(true);
    };

    const handleCloseModal = () => {
        setIsModalOpen(false);
        setSelectedProduct(null);
    };

    return (
        <section className="py-20 bg-gray-50 relative overflow-hidden" id="products">
            {/* Background Pattern */}
            <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIxMDAlIiBoZWlnaHQ9IjEwMCUiPjxkZWZzPjxwYXR0ZXJuIGlkPSJwYXR0ZXJuIiB3aWR0aD0iNDAiIGhlaWdodD0iNDAiIHBhdHRlcm5Vbml0cz0idXNlclNwYWNlT25Vc2UiIHBhdHRlcm5UcmFuc2Zvcm09InJvdGF0ZSg0NSkiPjxyZWN0IHg9IjAiIHk9IjAiIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0icmdiYSgxNiwgMTg1LDEyOSwwLjA1KSIvPjwvcGF0dGVybj48L2RlZnM+PHJlY3Qgd2lkdGg9IjEwMCUiIGhlaWdodD0iMTAwJSIgZmlsbD0idXJsKCNwYXR0ZXJuKSIvPjwvc3ZnPg==')] opacity-30"></div>
            
            {/* Decorative Elements */}
            <div className="absolute top-20 left-10 w-24 h-24 rounded-full bg-gradient-to-r from-green-300 to-emerald-300 opacity-20 blur-xl"></div>
            <div className="absolute bottom-20 right-10 w-32 h-32 rounded-full bg-gradient-to-r from-emerald-300 to-green-300 opacity-20 blur-xl"></div>
            
            <div className="container mx-auto px-6 relative z-10">
                {/* Section Header */}
                <motion.div 
                    className="text-center mb-16"
                    initial={{ opacity: 0, y: -20 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    transition={{ duration: 0.7 }}
                    viewport={{ once: true }}
                >
                    <h2 className="text-4xl font-bold text-gray-800 mb-4">Our Products</h2>
                    
                    {/* Divider */}
                    <div className="flex justify-center mb-6">
                        <div className="h-1 w-24 bg-gradient-to-r from-green-600 to-emerald-600 rounded-full"></div>
                    </div>
                    
                    <p className="text-lg text-gray-600 max-w-3xl mx-auto">
                        Discover our range of high-quality products designed to meet your needs with excellence and reliability.
                    </p>
                </motion.div>
                
                {/* Products Grid */}
                <motion.div 
                    className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6"
                    variants={containerVariants}
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true, amount: 0.2 }}
                >
                    {products.slice(0, 8).map((product) => (
                        <ProductCard key={product.id} product={product} onProductClick={handleProductClick} />
                    ))}
                </motion.div>
                
                {/* View All Products Button */}
                <motion.div 
                    className="mt-12 text-center"
                    initial={{ opacity: 0 }}
                    whileInView={{ opacity: 1 }}
                    transition={{ duration: 0.7, delay: 0.5 }}
                    viewport={{ once: true }}
                >
                    <button className="px-8 py-3 bg-white border-2 border-green-600 text-green-600 font-medium rounded-lg hover:bg-green-600 hover:text-white transition-all duration-300 hover:shadow-lg transform hover:-translate-y-0.5">
                        View All Products
                    </button>
                </motion.div>
            </div>
            
            {/* Add to Cart Modal */}
            <AddToCartModal
                product={selectedProduct}
                onClose={handleCloseModal}
            />
            
            {/* Toast Notification */}
            <AnimatePresence>
                {showToast && (
                    <motion.div
                        initial={{ y: 50, opacity: 0 }}
                        animate={{ y: 0, opacity: 1 }}
                        exit={{ y: 50, opacity: 0 }}
                        className="fixed bottom-6 right-6 bg-green-600 text-white px-6 py-3 rounded-lg shadow-lg flex items-center"
                    >
                        <FaShoppingCart className="mr-2" />
                        <span>Product added to cart!</span>
                    </motion.div>
                )}
            </AnimatePresence>
        </section>
    );
};

export default ProductsSection;