import React, { useState } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { FaBaby, FaCheckCircle } from 'react-icons/fa';
import SchoolCard from '@/Components/SchoolCard';
import SchoolModal from '@/Components/SchoolModal';
// --- Configuration ---
const containerVariants = {
    hidden: { opacity: 0 },
    visible: {
        opacity: 1,
        transition: {
            staggerChildren: 0.15,
            delayChildren: 0.3
        }
    }
};

// --- Main Component: PregnancySchoolsSection ---
const PregnancySchoolsSection = ({ schools }) => {
    const [selectedSchool, setSelectedSchool] = useState(null);
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [showToast, setShowToast] = useState(false);
    const [toastMessage, setToastMessage] = useState('');

    const handleOpenModal = (school) => {
        setSelectedSchool(school);
        setIsModalOpen(true);
    };

    const handleCloseModal = () => {
        setIsModalOpen(false);
    };

    const handleBook = (bookingData) => {
        console.log('Booking data:', bookingData);
        setToastMessage(
            bookingData.schoolId 
                ? `Your booking for "${bookingData.schoolName}" has been confirmed! We'll contact you soon.`
                : `Your registration for "${bookingData.schoolName}" has been confirmed! We'll contact you soon.`
        );
        setShowToast(true);
        setTimeout(() => setShowToast(false), 5000);
    };

    return (
        <section className="py-20 lg:py-32 bg-gradient-to-br from-green-50 to-emerald-50" id="pregnancy-schools">
            <div className="container mx-auto px-6">
                <motion.div 
                    className="text-center mb-16 lg:mb-24"
                    initial={{ opacity: 0, y: -30 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    transition={{ duration: 0.7 }}
                    viewport={{ once: true }}
                >
                    <div className="flex justify-center mb-4">
                        <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center">
                            <FaBaby className="text-2xl text-green-600" />
                        </div>
                    </div>
                    <p className="text-lg font-semibold text-green-600 mb-3">
                        Empowering Your Journey to Motherhood
                    </p>
                    <h2 className="text-5xl lg:text-6xl font-extrabold text-gray-800 mb-6">
                        Pregnancy Schools
                    </h2>
                    <div className="flex justify-center mb-6">
                        <div className="h-2 w-20 bg-gradient-to-r from-green-600 to-emerald-600 rounded-full"></div>
                    </div>
                    <p className="mt-8 text-xl text-gray-600 max-w-4xl mx-auto leading-relaxed">
                        Join our expert-led pregnancy schools to prepare for the beautiful journey of motherhood. 
                        Both free and premium programs available to suit your needs.
                    </p>
                </motion.div>
                
                {/* Schools grid */}
                <motion.div 
                    className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8"
                    variants={containerVariants}
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true, amount: 0.1 }}
                >
                    {schools && schools.length > 0 ? (
                        schools.map((school) => (
                            <SchoolCard key={school.id} school={school} onOpenModal={handleOpenModal} />
                        ))
                    ) : (
                        <div className="col-span-full text-center py-10">
                            <p className="text-xl text-gray-600">No pregnancy schools available at the moment.</p>
                        </div>
                    )}
                </motion.div>
            </div>
            
            {/* School Modal */}
            <SchoolModal
                school={selectedSchool}
                isOpen={isModalOpen}
                onClose={handleCloseModal}
                onBook={handleBook}
            />
            
            {/* Toast Notification */}
            <AnimatePresence>
                {showToast && (
                    <motion.div
                        initial={{ y: 50, opacity: 0 }}
                        animate={{ y: 0, opacity: 1 }}
                        exit={{ y: 50, opacity: 0 }}
                        className="fixed bottom-6 right-6 bg-green-600 text-white px-6 py-3 rounded-lg shadow-lg flex items-center"
                    >
                        <FaCheckCircle className="mr-2" />
                        <span>{toastMessage}</span>
                    </motion.div>
                )}
            </AnimatePresence>
        </section>
    );
};

export default PregnancySchoolsSection;