import { motion } from 'framer-motion';
import { 
    IoCodeSlash, 
    IoPhonePortraitOutline, 
    IoBarChartOutline, 
    IoSparkles, 
    IoCloudOutline, 
    IoBusinessOutline,
    IoArrowForward,
    IoPricetagOutline
} from 'react-icons/io5';

const itemVariants = {
    hidden: { y: 30, opacity: 0, scale: 0.95 },
    visible: { 
        y: 0, 
        opacity: 1, 
        scale: 1, 
        transition: { type: "spring", stiffness: 150, damping: 20, duration: 0.6 } 
    }
};

// Icon mapping based on service type or ID
const getIconForService = (service) => {
    const iconMap = {
        1: IoCodeSlash,             // Web Development
        2: IoPhonePortraitOutline,  // Mobile App
        3: IoBarChartOutline,       // Digital Marketing
        4: IoSparkles,              // AI & Automation
        5: IoCloudOutline,          // Cloud Infrastructure
        6: IoBusinessOutline        // Enterprise Consulting
    };
    
    return iconMap[service.id] || IoSparkles;
};

const ServiceCard = ({ service, onBookService }) => {
    const ACCENT_COLOR = 'from-green-500 to-green-700';
    
    // Debug: Log the image URL
    console.log('Service image URL:', service);
    
    // Function to handle image loading errors
    const handleImageError = (e) => {
        console.error('Image failed to load:', e.target.src);
        e.target.onerror = null;
        e.target.src = "https://placehold.co/600x400/10b981/ffffff?text=Image+Loading+Error";
    };
    
    return (
        <motion.div 
            className="group relative bg-white rounded-3xl overflow-hidden shadow-xl border border-gray-100 transition-all duration-500 
                     hover:shadow-3xl hover:shadow-green-200/50 hover:-translate-y-2 cursor-pointer
                     flex flex-col h-full"
            variants={itemVariants}
            whileHover={{ scale: 1.02, rotate: 0.5 }}
            transition={{ type: "spring", stiffness: 300 }}
        >
            {/* Image Area */}
            <div className="relative h-56 overflow-hidden">
                <img 
                    src={service.image_url}
                    alt={service.title}
                    className="w-full h-full object-cover transition-transform duration-700 brightness-95 group-hover:scale-110 group-hover:brightness-100"
                />
                
                {/* Subtle gradient overlay */}
                <div className="absolute inset-0 bg-gradient-to-t from-gray-900/30 via-transparent to-transparent"></div>
                
                {/* Price Badge */}
                <motion.div 
                    className={`absolute top-4 right-4 bg-gradient-to-r ${ACCENT_COLOR} rounded-full px-4 py-2 shadow-lg flex items-center`}
                    whileHover={{ scale: 1.05 }}
                    transition={{ type: "spring", stiffness: 400 }}
                >
                    <IoPricetagOutline className="text-white mr-2" />
                    <span className="font-bold text-white text-sm">{service.price ? `GHS ${service.price}` : 'Free'}</span>
                </motion.div>
            </div>
            
            {/* Content Area */}
            <div className="p-6 flex-grow flex flex-col">
                <h3 className="text-xl font-extrabold text-gray-800 tracking-tight transition-all duration-300 relative">
                    <span className="relative z-10 group-hover:text-green-700 inline-block">
                        {service.title}
                        {/* Animated underline */}
                        <motion.span 
                            className="absolute bottom-0 left-0 h-0.5 bg-gradient-to-r from-green-500 to-green-700 rounded-full transform origin-left"
                            initial={{ scaleX: 0 }}
                            whileHover={{ scaleX: 1 }}
                            transition={{ duration: 0.3, ease: "easeInOut" }}
                            style={{ width: "100%" }}
                        ></motion.span>
                    </span>
                </h3>
                
                <p className="text-gray-500 mb-4 line-clamp-3 text-base flex-grow">
                    {service.description}
                </p>
                
                {/* Service Features (Placeholder for now, as service model doesn't have features) */}
                <div className="mb-6">
                    <ul className="space-y-2">
                        {/* {service.features && service.features.slice(0, 3).map((feature, index) => (
                            <motion.li 
                                key={index} 
                                className="flex items-start text-sm text-gray-700"
                                initial={{ opacity: 0, x: -10 }}
                                animate={{ opacity: 1, x: 0 }}
                                transition={{ delay: index * 0.05 }}
                            >
                                <IoCheckmarkCircle className="text-green-500 mt-1 mr-2 flex-shrink-0" />
                                <span className="line-clamp-1">{feature}</span>
                            </motion.li>
                        ))} */}
                    </ul>
                </div>

                {/* Book Service Button */}
                <motion.button
                    className={`w-full py-4 px-4 rounded-xl font-bold text-lg text-white 
                                 bg-gradient-to-r ${ACCENT_COLOR} hover:from-green-600 hover:to-green-800
                                 shadow-lg hover:shadow-xl transition-all duration-300`}
                    whileHover={{ scale: 1.03, boxShadow: '0 10px 20px rgba(16, 185, 129, 0.4)' }}
                    whileTap={{ scale: 0.98 }}
                    onClick={() => onBookService && onBookService(service)}
                >
                    <span className="flex items-center justify-center">
                        <IoArrowForward className="mr-2 h-5 w-5" />
                        Explore & Book
                    </span>
                </motion.button>
                
                {/* Provider Info (Optional) */}
                {/* {service.provider && (
                    <div className="flex items-center justify-center mt-4 text-xs text-gray-400">
                        <IoPersonOutline className="mr-1" />
                        <span>Provided by {service.provider}</span>
                    </div>
                )} */}
            </div>
        </motion.div>
    );
};

export default ServiceCard;