import React, { useState } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { 
    IoCheckmarkCircleOutline,
    IoInformationCircleOutline,
} from 'react-icons/io5';
import BookingModal from './BookingModal';
import ServiceCard from './ServiceCard';

// --- Configuration & Constants ---
const ACCENT_COLOR = 'from-green-500 to-green-700'; 
const ACCENT_TEXT_COLOR = 'text-green-600';

// --- Shared Framer Motion Variants ---
const containerVariants = {
    hidden: { opacity: 0 },
    visible: { 
        opacity: 1, 
        transition: { 
            staggerChildren: 0.15, 
            delayChildren: 0.2 
        } 
    }
};

// --- Toast Notification Component ---
const Toast = ({ message, color, visible }) => {
    const iconMap = {
        green: IoCheckmarkCircleOutline,
        blue: IoInformationCircleOutline,
    };
    const Icon = iconMap[color] || IoInformationCircleOutline;

    const colorClasses = {
        green: 'bg-green-500 text-white shadow-green-400/50',
        blue: 'bg-indigo-500 text-white shadow-indigo-400/50',
    };

    return (
        <AnimatePresence>
            {visible && (
                <motion.div
                    initial={{ y: 50, opacity: 0, scale: 0.9 }}
                    animate={{ y: 0, opacity: 1, scale: 1 }}
                    exit={{ y: 50, opacity: 0, scale: 0.9 }}
                    transition={{ type: "spring", stiffness: 300, damping: 30 }}
                    className={`fixed bottom-6 right-6 p-4 rounded-xl shadow-2xl z-50 flex items-center min-w-80 ${colorClasses[color]}`}
                >
                    <Icon className="text-2xl mr-3 flex-shrink-0" />
                    <span className="font-semibold text-base">{message}</span>
                </motion.div>
            )}
        </AnimatePresence>
    );
};

// --- Main Services Section Component ---
const ServicesSection = ({ services }) => {
    const [toast, setToast] = useState({ visible: false, message: '', color: 'blue' });
    const [selectedService, setSelectedService] = useState(null);
    const [isModalOpen, setIsModalOpen] = useState(false);

    const showToast = (message, color = 'blue') => {
        // Clear any existing timeout
        setToast({ visible: true, message, color });
        
        // Hide after 3 seconds
        setTimeout(() => {
            setToast(prev => ({ ...prev, visible: false }));
        }, 3000); 
    };

    // Open modal with selected service
    const handleOpenModal = (service) => {
        setSelectedService(service);
        setIsModalOpen(true);
    };

    // Close modal
    const handleCloseModal = () => {
        setIsModalOpen(false);
    };

    // Handle booking submission
    const handleBookService = (bookingData) => {
        console.log(`Booking service: ${selectedService.title}`, bookingData);
        showToast(`Your booking for '${selectedService.title}' has been confirmed! We'll contact you soon.`, 'green');
    };

    return (
        <div className="min-h-screen bg-white font-sans antialiased">
            <style>
                {/* Ensure Inter font is used via Tailwind config */}
                {`html { scroll-behavior: smooth; }`}
            </style>
            
            {/* Simplified Header */}
            <header className="pt-24 pb-16 bg-gray-50">
                <div className="container mx-auto px-4 sm:px-6 max-w-7xl">
                    <motion.div 
                        className="text-center"
                        initial={{ opacity: 0, y: -20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.6 }}
                    >
                        {/* Simple Badge */}
                        <div className="inline-block px-4 py-2 mb-6 bg-green-50 rounded-full">
                            <span className={`text-sm font-semibold uppercase tracking-widest ${ACCENT_TEXT_COLOR}`}>
                                Core Capabilities
                            </span>
                        </div>
                        
                        {/* Clean Heading */}
                        <h1 className="text-4xl md:text-5xl lg:text-6xl font-bold text-gray-900 mb-6 leading-tight">
                            Our <span className={`bg-gradient-to-r ${ACCENT_COLOR} bg-clip-text text-transparent`}>Service Matrix</span>
                        </h1>
                        
                        {/* Clear Description */}
                        <p className="text-lg text-gray-600 max-w-3xl mx-auto">
                            Hyper-focused, integrated solutions engineered for future-proof digital transformation and measurable business growth.
                        </p>
                    </motion.div>
                </div>
            </header>
            
            <section className="py-20 bg-white" id="services"> 
                <div className="container mx-auto px-4 sm:px-6 max-w-7xl">
                    {/* Services Grid with staggered animation */}
                    <motion.div 
                        className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-10"
                        variants={containerVariants}
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true, amount: 0.1 }}
                    >
                        {services && services.length > 0 ? (
                            services.map((service) => (
                                <ServiceCard 
                                    key={service.id}
                                    service={service}
                                    onBookService={handleOpenModal}
                                />
                            ))
                        ) : (
                            <div className="col-span-full text-center py-10">
                                <p className="text-2xl text-gray-500">No services available at the moment.</p>
                            </div>
                        )}
                    </motion.div>
                </div>
            </section>
            
            {/* Service Details Modal */}
            <BookingModal
                service={selectedService}
                show={isModalOpen}
                onClose={handleCloseModal}
                onBookService={handleBookService}
            />
            
            <Toast 
                message={toast.message} 
                color={toast.color} 
                visible={toast.visible} 
            />
        </div>
    );
};

export default ServicesSection;