import React from 'react';
import { motion } from 'framer-motion';
import { FaLinkedin, FaTwitter, FaEnvelope, FaGlobe } from 'react-icons/fa';

// Animation variants
const containerVariants = {
    hidden: { opacity: 0 },
    visible: {
        opacity: 1,
        transition: {
            staggerChildren: 0.2,
            delayChildren: 0.3
        }
    }
};

const itemVariants = {
    hidden: { y: 30, opacity: 0, scale: 0.9 },
    visible: {
        y: 0,
        opacity: 1,
        scale: 1,
        transition: { duration: 0.6, ease: "easeOut" }
    }
};

const TeamMemberCard = ({ member }) => {
    return (
        <motion.div 
            className="relative group"
            variants={itemVariants}
            whileHover={{ 
                y: -15,
                transition: {
                    type: "spring",
                    stiffness: 300,
                    damping: 15
                }
            }}
        >
            {/* Cozy frame decoration */}
            <div className="absolute -inset-2 bg-gradient-to-r from-red-600 to-yellow-500 rounded-3xl opacity-20 group-hover:opacity-30 transition-all duration-400 blur-sm"></div>
            
            {/* Main card */}
            <div className="relative bg-white rounded-2xl shadow-lg overflow-hidden border border-amber-100/50 transition-all duration-300 group-hover:shadow-xl">
                {/* Image with cozy frame effect */}
                <div className="relative">
                    {/* Decorative corner elements */}
                    <div className="absolute top-0 left-0 w-12 h-12 border-t-3 border-l-3 border-red-400 rounded-tl-xl opacity-60 z-10"></div>
                    <div className="absolute top-0 right-0 w-12 h-12 border-t-3 border-r-3 border-orange-400 rounded-tr-xl opacity-60 z-10"></div>
                    
                    {/* Profile image with cozy frame - now larger and rectangular */}
                    <div className="relative w-full h-64 overflow-hidden">
                        {/* Animated frame border */}
                        <motion.div 
                            className="absolute inset-0 border-4 border-white shadow-md"
                            whileHover={{ 
                                scale: 1.02,
                                transition: {
                                    type: "spring",
                                    stiffness: 400,
                                    damping: 10
                                }
                            }}
                        ></motion.div>
                        
                        {/* Image with bounce effect - now larger and rectangular */}
                        <motion.img 
                            src={member.image_url} 
                            alt={member.name} 
                            className="bg-cover bg-no-repeat"
                            whileHover={{ 
                                scale: 1.05,
                                transition: {
                                    type: "spring",
                                    stiffness: 400,
                                    damping: 10
                                }
                            }}
                        />
                        
                        {/* Decorative overlay */}
                        <div className="absolute inset-0 bg-gradient-to-t from-black/20 to-transparent"></div>
                        
                        {/* Floating decoration */}
                        <motion.div 
                            className="absolute top-4 right-4 w-10 h-10 rounded-full bg-amber-100 flex items-center justify-center text-amber-500 shadow-md z-10"
                            whileHover={{ 
                                rotate: 15,
                                scale: 1.2,
                                transition: {
                                    type: "spring",
                                    stiffness: 500,
                                    damping: 10
                                }
                            }}
                        >
                            <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                <path fillRule="evenodd" d="M3.172 5.172a4 4 0 015.656 0L10 6.343l1.172-1.171a4 4 0 115.656 5.656L10 17.657l-6.828-6.829a4 4 0 010-5.656z" clipRule="evenodd" />
                            </svg>
                        </motion.div>
                    </div>
                    
                    {/* Cozy bottom decoration for image */}
                    <div className="absolute bottom-0 left-0 w-full h-1 bg-gradient-to-r from-red-600 to-yellow-500"></div>
                </div>
                
                {/* Content section */}
                <div className="p-6">
                    {/* Name and position */}
                    <h3 className="text-xl font-bold text-gray-800 mb-1">{member.name}</h3>
                    
                    {/* Department badge with bounce effect */}
                    <motion.div 
                        className="inline-block px-4 py-2 mb-4 text-sm font-semibold rounded-full bg-gradient-to-r from-red-600 to-yellow-500 text-white shadow-sm"
                        whileHover={{ 
                            scale: 1.1,
                            transition: {
                                type: "spring",
                                stiffness: 500,
                                damping: 10
                            }
                        }}
                    >
                        {member.position}
                    </motion.div>
                    
                    {/* Social links with bounce effect */}
                </div>
                
                {/* Cozy bottom decoration */}
                <div className="absolute bottom-0 left-0 w-full h-1 bg-gradient-to-r from-red-600 to-yellow-500"></div>
            </div>
            
            {/* Cozy ornament */}
            <div className="absolute -bottom-4 left-1/2 transform -translate-x-1/2 w-12 h-1 bg-gradient-to-r from-red-600 to-yellow-500 rounded-full"></div>
            <div className="absolute -bottom-6 left-1/2 transform -translate-x-1/2 w-8 h-1 bg-gradient-to-r from-red-600 to-yellow-500 rounded-full"></div>
        </motion.div>
    );
};

const TeamSection = ({ teams }) => {
    return (
        <section className="py-24 bg-gradient-to-b from-amber-50 to-white relative overflow-hidden" id="team">
            {/* Cozy background pattern */}
            <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIxMDAlIiBoZWlnaHQ9IjEwMCUiPjxkZWZzPjxwYXR0ZXJuIGlkPSJwYXR0ZXJuIiB3aWR0aD0iNDAiIGhlaWdodD0iNDAiIHBhdHRlcm5Vbml0cz0idXNlclNwYWNlT25Vc2UiIHBhdHRlcm5UcmFuc2Zvcm09InJvdGF0ZSg0NSkiPjxyZWN0IHg9IjAiIHk9IjAiIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0icmdiYSgyNTUsMjM2LDE3OSwwLjA1KSIvPjwvcGF0dGVybj48L2RlZnM+PHJlY3Qgd2lkdGg9IjEwMCUiIGhlaWdodD0iMTAwJSIgZmlsbD0idXJsKCNwYXR0ZXJuKSIvPjwvc3ZnPg==')] opacity-30"></div>
            
            {/* Decorative elements */}
            <div className="absolute top-20 left-10 w-24 h-24 rounded-full bg-gradient-to-r from-amber-300 to-orange-300 opacity-20 blur-xl"></div>
            <div className="absolute bottom-20 right-10 w-32 h-32 rounded-full bg-gradient-to-r from-orange-300 to-amber-300 opacity-20 blur-xl"></div>
            
            <div className="container mx-auto px-6 relative z-10">
                {/* Section header with cozy styling */}
                <motion.div 
                    className="text-center mb-20"
                    initial={{ opacity: 0, y: -20 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    transition={{ duration: 0.7 }}
                    viewport={{ once: true }}
                >
                    {/* Cozy ornament above title */}
                    <div className="flex justify-center mb-6">
                        <div className="w-24 h-1 bg-gradient-to-r from-red-600 to-yellow-500 rounded-full"></div>
                        <div className="w-4 h-4 mx-2 rounded-full bg-gradient-to-r from-red-600 to-yellow-500"></div>
                        <div className="w-24 h-1 bg-gradient-to-r from-red-600 to-yellow-500 rounded-full"></div>
                    </div>
                    
                    <h2 className="text-5xl font-bold text-gray-800 mb-4 relative inline-block">
                        <span className="relative z-10">Meet Our Team</span>
                        <span className="absolute bottom-0 left-0 w-full h-3 bg-gradient-to-r from-amber-300/30 to-orange-300/30 rounded-full -z-10"></span>
                    </h2>
                    
                    {/* Cozy divider */}
                    <div className="flex justify-center mt-6">
                        <div className="h-1 w-32 bg-gradient-to-r from-red-600 to-yellow-500 rounded-full"></div>
                    </div>
                    
                    <p className="mt-8 text-xl text-gray-600 max-w-3xl mx-auto">
                        The friendly faces behind our success, combining expertise, passion, and a warm approach to drive EK Company forward.
                    </p>
                </motion.div>
                
                {/* Team members grid - Updated to display 4 cards per row */}
                <motion.div 
                    className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6"
                    variants={containerVariants}
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true, amount: 0.2 }}
                >
                    {teams.slice(0, 4).map((member) => (
                        <TeamMemberCard key={member.id} member={member} />
                    ))}
                </motion.div>
                
                {/* Cozy quote at the bottom */}
                <motion.div 
                    className="mt-24 text-center max-w-3xl mx-auto p-8 bg-white rounded-2xl border border-amber-100 shadow-lg relative"
                    initial={{ opacity: 0, y: 20 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    transition={{ duration: 0.7, delay: 0.5 }}
                    viewport={{ once: true }}
                    whileHover={{ 
                        y: -10,
                        transition: {
                            type: "spring",
                            stiffness: 300,
                            damping: 15
                        }
                    }}
                >
                    {/* Cozy quote marks */}
                    <div className="absolute top-0 left-0 transform -translate-y-1/2 -translate-x-1/2 text-6xl text-red-400 opacity-20">"</div>
                    <div className="absolute bottom-0 right-0 transform translate-y-1/2 translate-x-1/2 text-6xl text-orange-400 opacity-20">"</div>
                    
                    <p className="text-xl italic text-gray-700 mb-6">
                        "Alone we can do so little; together we can do so much. The strength of the team is each individual member. The strength of each member is the team."
                    </p>
                    <p className="text-gray-500">— Helen Keller</p>
                </motion.div>
            </div>
        </section>
    );
};

export default TeamSection;