import React from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link } from '@inertiajs/react';
import { formatDistanceToNow } from 'date-fns';
import { 
    ChatBubbleLeftRightIcon, 
    UserGroupIcon, 
    ClockIcon,
    CheckCircleIcon,
    ArrowTopRightOnSquareIcon
} from '@heroicons/react/24/outline';

// --- Helper function to calculate analytics ---
const getAnalytics = (conversations) => {
    const totalConversations = conversations.length;
    const oneDayAgo = new Date();
    oneDayAgo.setDate(oneDayAgo.getDate() - 1);
    const activeConversations = conversations.filter(conv => new Date(conv.last_message_at) > oneDayAgo).length;
    
    // This is a placeholder. In a real app, you'd track unread messages per user.
    const unreadCount = conversations.filter(conv => conv.has_unread).length; 

    return { totalConversations, activeConversations, unreadCount };
};

// --- Sub-Component: Analytics Card ---
const AnalyticsCard = ({ title, value, icon, color }) => {
    return (
        <div className="bg-white overflow-hidden shadow-lg rounded-lg border border-gray-100">
            <div className="p-6">
                <div className="flex items-center">
                    <div className={`flex-shrink-0 p-3 rounded-lg ${color}`}>
                        {icon}
                    </div>
                    <div className="ml-5 w-0 flex-1">
                        <dl>
                            <dt className="text-sm font-medium text-gray-500 truncate uppercase tracking-wide">{title}</dt>
                            <dd className="text-2xl font-bold text-gray-900">{value}</dd>
                        </dl>
                    </div>
                </div>
            </div>
        </div>
    );
};

// --- Sub-Component: Conversation Card ---
const ConversationCard = ({ conversation }) => {
    const isActive = new Date(conversation.last_message_at) > new Date(Date.now() - 24 * 60 * 60 * 1000);
    
    return (
        <Link
            href={route('admin.conversations.show', conversation.id)}
            className="group block p-4 bg-white border border-gray-200 rounded-lg shadow-sm hover:shadow-md hover:border-emerald-300 transition-all duration-200"
        >
            <div className="flex items-center space-x-4">
                {/* Status Indicator */}
                <div className="flex-shrink-0">
                    {isActive ? (
                        <div className="h-3 w-3 bg-emerald-500 rounded-full animate-pulse" title="Active"></div>
                    ) : (
                        <div className="h-3 w-3 bg-gray-300 rounded-full" title="Inactive"></div>
                    )}
                </div>

                {/* Avatars */}
                <div className="flex-shrink-0 flex -space-x-2">
                    <div className="h-10 w-10 rounded-full bg-gray-300 flex items-center justify-center ring-2 ring-white">
                        <span className="text-sm font-medium leading-none text-gray-600">
                            {conversation.user?.name?.charAt(0).toUpperCase()}
                        </span>
                    </div>
                    <div className="h-10 w-10 rounded-full bg-emerald-500 flex items-center justify-center ring-2 ring-white">
                        <span className="text-sm font-medium leading-none text-white">
                            {conversation.doctor?.name?.charAt(0).toUpperCase()}
                        </span>
                    </div>
                </div>

                {/* Content */}
                <div className="flex-1 min-w-0">
                    <p className="text-sm font-semibold text-gray-900 truncate group-hover:text-emerald-600">
                        {conversation.subject || 'No Subject'}
                    </p>
                    <p className="text-xs text-gray-500 truncate">
                        {conversation.user?.name} & {conversation.doctor?.name}
                    </p>
                    <p className="text-xs text-gray-400 mt-1">
                        {conversation.last_message_at && formatDistanceToNow(new Date(conversation.last_message_at), { addSuffix: true })}
                    </p>
                </div>

                {/* Action Icon */}
                <div className="flex-shrink-0 self-center">
                    <ArrowTopRightOnSquareIcon className="h-5 w-5 text-gray-400 group-hover:text-emerald-600" />
                </div>
            </div>
        </Link>
    );
};

export default function Index({ auth, conversations }) {
    const analytics = getAnalytics(conversations.data);

    return (
        <AdminLayout
            user={auth.user}
            header={
                <div>
                    <h2 className="font-semibold text-2xl text-gray-800 leading-tight">Conversations Dashboard</h2>
                    <p className="mt-1 text-sm text-gray-600">Monitor and manage all communications between users and doctors.</p>
                </div>
            }
        >
            <Head title="Conversations" />

            <div className="py-6">
                <div className="mx-auto sm:px-6 lg:px-8">
                    {/* --- Analytics Section --- */}
                    <div className="grid grid-cols-1 gap-5 sm:grid-cols-2 lg:grid-cols-3 mb-8">
                        <AnalyticsCard
                            title="Total Conversations"
                            value={analytics.totalConversations}
                            icon={<ChatBubbleLeftRightIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-gradient-to-r from-emerald-500 to-emerald-600"
                        />
                        <AnalyticsCard
                            title="Active (24h)"
                            value={analytics.activeConversations}
                            icon={<UserGroupIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-gradient-to-r from-blue-500 to-blue-600"
                        />
                        <AnalyticsCard
                            title="Unread Messages"
                            value={analytics.unreadCount}
                            icon={<ClockIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-gradient-to-r from-yellow-500 to-yellow-600"
                        />
                    </div>

                    {/* --- Conversations List --- */}
                    <div className="bg-white shadow-xl rounded-lg overflow-hidden">
                        <div className="px-6 py-4 border-b border-gray-200">
                            <h3 className="text-lg font-medium text-gray-900">Recent Conversations</h3>
                        </div>
                        
                        <div className="divide-y divide-gray-200">
                            {conversations.data.length > 0 ? (
                                conversations.data.map((conversation) => (
                                    <ConversationCard key={conversation.id} conversation={conversation} />
                                ))
                            ) : (
                                // Beautiful Empty State
                                <div className="text-center py-12">
                                    <ChatBubbleLeftRightIcon className="mx-auto h-12 w-12 text-gray-400" />
                                    <h3 className="mt-2 text-sm font-medium text-gray-900">No conversations</h3>
                                    <p className="mt-1 text-sm text-gray-500">When a user starts a conversation, it will appear here.</p>
                                </div>
                            )}
                        </div>
                    </div>

                    {/* --- Pagination --- */}
                    {conversations.links && conversations.links.length > 3 && (
                        <div className="mt-6 flex justify-center">
                            {conversations.links.map((link, key) => (
                                <Link
                                    key={key}
                                    href={link.url || '#'}
                                    className={`mr-1 mb-1 px-4 py-3 text-sm leading-4 border rounded-md transition-colors duration-150 ${
                                        link.active
                                            ? 'bg-emerald-600 text-white border-emerald-600'
                                            : 'bg-white text-gray-700 border-gray-300 hover:bg-gray-50'
                                    } ${!link.url ? 'opacity-50 cursor-not-allowed' : ''}`}
                                    dangerouslySetInnerHTML={{ __html: link.label }}
                                />
                            ))}
                        </div>
                    )}
                </div>
            </div>
        </AdminLayout>
    );
}