import React, { useState } from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, useForm, Link } from '@inertiajs/react';
import InputLabel from '@/Components/InputLabel';
import TextInput from '@/Components/TextInput';
import InputError from '@/Components/InputError';
import PrimaryButton from '@/Components/PrimaryButton';
import {
    ArrowLeftIcon,
    UserCircleIcon, // General user/person icon
    EnvelopeIcon,   // Email icon
    PhoneIcon,      // Mobile number icon
    LockClosedIcon, // Password icon
    PhotoIcon,      // Image upload icon
    XMarkIcon,      // Close/remove icon
    PlusCircleIcon, // Create/add icon
    ClipboardDocumentListIcon // For general doctor/medical info
} from '@heroicons/react/24/outline';

// --- Reusable Component: FormSection (for clear grouping) ---
const FormSection = ({ title, description, children, icon: IconComponent }) => (
    <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 lg:gap-8 border-t border-gray-200 pt-8">
        <div className="lg:col-span-1">
            <h3 className="text-xl font-bold text-gray-800 flex items-center">
                {IconComponent && <IconComponent className="w-6 h-6 mr-2 text-teal-500" />}
                {title}
            </h3>
            <p className="mt-2 text-sm text-gray-500">{description}</p>
        </div>
        <div className="lg:col-span-2 space-y-6">
            {children}
        </div>
    </div>
);

export default function Create({ auth }) {
    const { data, setData, post, processing, errors, reset } = useForm({
        name: '',
        email: '',
        password: '',
        password_confirmation: '',
        mobile_number: '',
        profile_picture: null, // Added for image upload
    });

    const [imagePreview, setImagePreview] = useState(null);

    const handleImageChange = (e) => {
        const file = e.target.files[0];
        if (file) {
            setData('profile_picture', file);

            const reader = new FileReader();
            reader.onload = (e) => {
                setImagePreview(e.target.result);
            };
            reader.readAsDataURL(file);
        }
    };

    const handleRemoveImage = () => {
        setData('profile_picture', null);
        setImagePreview(null);
        // Clear the file input visually
        document.getElementById('profile_picture').value = '';
    };

    const submit = (e) => {
        e.preventDefault();
        post(route('admin.doctors.store'), {
            forceFormData: true, // Essential for file uploads
        });
    };

    return (
        <AdminLayout
            user={auth.user}
            // Removed default header to use a more prominent one inside the main content area
            // header={<h2 className="font-semibold text-xl text-gray-800 leading-tight">Add New Doctor</h2>}
        >
            <Head title="Add New Doctor" />

            <div className="py-8">
                <div className="mx-auto sm:px-6 lg:px-8">
                    
                    {/* --- Page Header --- */}
                    <div className="mb-6 flex flex-col md:flex-row md:items-center md:justify-between">
                        <div className="flex-1 min-w-0">
                            <Link 
                                href={route('admin.doctors.index')} 
                                className="inline-flex items-center text-sm font-medium text-gray-500 hover:text-teal-600 transition duration-150"
                            >
                                <ArrowLeftIcon className="-ml-0.5 mr-1 h-5 w-5" aria-hidden="true" />
                                Back to Doctors List
                            </Link>
                            <h1 className="mt-2 text-3xl font-extrabold leading-7 text-gray-900 sm:text-4xl sm:truncate">
                                **Register a New Doctor** 🧑‍⚕️
                            </h1>
                        </div>
                    </div>

                    {/* --- Main Form Card --- */}
                    <div className="bg-white shadow-2xl sm:rounded-xl overflow-hidden border border-gray-100">
                        <form onSubmit={submit} className="p-6 sm:p-10 space-y-10">
                            
                            {/* Section 1: Personal Details */}
                            <FormSection 
                                title="Doctor's Personal Information"
                                description="Enter the full name and contact details for the new doctor."
                                icon={UserCircleIcon}
                            >
                                <div>
                                    <InputLabel htmlFor="name" value="Full Name" />
                                    <div className="mt-1 relative rounded-lg shadow-sm">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <UserCircleIcon className="h-5 w-5 text-gray-400" aria-hidden="true" />
                                        </div>
                                        <TextInput
                                            id="name"
                                            type="text"
                                            name="name"
                                            value={data.name}
                                            placeholder="e.g., Dr. Alice Johnson"
                                            className="mt-1 block w-full pl-10 rounded-lg border-gray-300 focus:border-teal-500 focus:ring-teal-500 shadow-sm"
                                            onChange={(e) => setData('name', e.target.value)}
                                            required
                                            isFocused={true}
                                        />
                                    </div>
                                    <InputError message={errors.name} className="mt-2" />
                                </div>

                                <div>
                                    <InputLabel htmlFor="email" value="Email Address" />
                                    <div className="mt-1 relative rounded-lg shadow-sm">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <EnvelopeIcon className="h-5 w-5 text-gray-400" aria-hidden="true" />
                                        </div>
                                        <TextInput
                                            id="email"
                                            type="email"
                                            name="email"
                                            value={data.email}
                                            placeholder="e.g., alice.johnson@clinic.com"
                                            className="mt-1 block w-full pl-10 rounded-lg border-gray-300 focus:border-teal-500 focus:ring-teal-500 shadow-sm"
                                            onChange={(e) => setData('email', e.target.value)}
                                            required
                                        />
                                    </div>
                                    <InputError message={errors.email} className="mt-2" />
                                </div>

                                <div>
                                    <InputLabel htmlFor="mobile_number" value="Mobile Number" />
                                    <div className="mt-1 relative rounded-lg shadow-sm">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <PhoneIcon className="h-5 w-5 text-gray-400" aria-hidden="true" />
                                        </div>
                                        <TextInput
                                            id="mobile_number"
                                            type="text" // Use text for phone numbers to allow various formats
                                            name="mobile_number"
                                            value={data.mobile_number}
                                            placeholder="e.g., +1 (555) 123-4567"
                                            className="mt-1 block w-full pl-10 rounded-lg border-gray-300 focus:border-teal-500 focus:ring-teal-500 shadow-sm"
                                            onChange={(e) => setData('mobile_number', e.target.value)}
                                        />
                                    </div>
                                    <InputError message={errors.mobile_number} className="mt-2" />
                                </div>
                            </FormSection>

                            {/* Section 2: Security & Access */}
                            <FormSection 
                                title="Account Security"
                                description="Set a secure password for the doctor's administrative account."
                                icon={LockClosedIcon}
                            >
                                <div>
                                    <InputLabel htmlFor="password" value="Password" />
                                    <div className="mt-1 relative rounded-lg shadow-sm">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <LockClosedIcon className="h-5 w-5 text-gray-400" aria-hidden="true" />
                                        </div>
                                        <TextInput
                                            id="password"
                                            type="password"
                                            name="password"
                                            value={data.password}
                                            className="mt-1 block w-full pl-10 rounded-lg border-gray-300 focus:border-teal-500 focus:ring-teal-500 shadow-sm"
                                            onChange={(e) => setData('password', e.target.value)}
                                            required
                                        />
                                    </div>
                                    <InputError message={errors.password} className="mt-2" />
                                </div>

                                <div>
                                    <InputLabel htmlFor="password_confirmation" value="Confirm Password" />
                                    <div className="mt-1 relative rounded-lg shadow-sm">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <LockClosedIcon className="h-5 w-5 text-gray-400" aria-hidden="true" />
                                        </div>
                                        <TextInput
                                            id="password_confirmation"
                                            type="password"
                                            name="password_confirmation"
                                            value={data.password_confirmation}
                                            className="mt-1 block w-full pl-10 rounded-lg border-gray-300 focus:border-teal-500 focus:ring-teal-500 shadow-sm"
                                            onChange={(e) => setData('password_confirmation', e.target.value)}
                                            required
                                        />
                                    </div>
                                    <InputError message={errors.password_confirmation} className="mt-2" />
                                </div>
                            </FormSection>

                            {/* Section 3: Profile Picture */}
                            <FormSection 
                                title="Profile Picture"
                                description="Upload a professional headshot for the doctor's profile."
                                icon={PhotoIcon}
                            >
                                {imagePreview ? (
                                    /* Image Preview and Remove Button */
                                    <div className="relative w-full sm:w-2/3 md:w-1/2 lg:w-1/3 rounded-xl overflow-hidden border border-teal-400 shadow-lg ring-2 ring-teal-300 ring-offset-2">
                                        <img 
                                            src={imagePreview} 
                                            alt="Profile Preview" 
                                            className="h-48 w-full object-cover" 
                                        />
                                        <button
                                            type="button"
                                            onClick={handleRemoveImage}
                                            className="absolute top-2 right-2 bg-red-600 text-white rounded-full p-1.5 hover:bg-red-700 transition-colors shadow-xl z-10"
                                            title="Remove profile picture"
                                        >
                                            <XMarkIcon className="h-5 w-5" />
                                        </button>
                                    </div>
                                ) : (
                                    /* File Input/Drop Area */
                                    <div className="mt-2 flex justify-center px-6 pt-5 pb-6 border-2 border-teal-300 border-dashed rounded-lg hover:border-teal-400 transition-colors bg-teal-50/50">
                                        <div className="space-y-1 text-center">
                                            <PhotoIcon className="mx-auto h-12 w-12 text-teal-500" />
                                            <div className="flex text-sm text-gray-600">
                                                <label
                                                    htmlFor="profile_picture"
                                                    className="relative cursor-pointer bg-white rounded-md font-medium text-teal-600 hover:text-teal-700 focus-within:outline-none focus-within:ring-2 focus-within:ring-offset-2 focus-within:ring-teal-500"
                                                >
                                                    <span>Upload a file</span>
                                                    <input 
                                                        id="profile_picture" 
                                                        name="profile_picture" 
                                                        type="file" 
                                                        className="sr-only" 
                                                        onChange={handleImageChange} 
                                                        accept="image/*"
                                                    />
                                                </label>
                                                <p className="pl-1 text-gray-500">or drag and drop</p>
                                            </div>
                                            <p className="text-xs text-gray-500">Square PNG, JPG recommended. Max 5MB.</p>
                                        </div>
                                    </div>
                                )}
                                <InputError message={errors.profile_picture} className="mt-2" />
                            </FormSection>

                            {/* --- Action Buttons --- */}
                            <div className="flex items-center justify-end space-x-4 pt-8 border-t border-gray-200">
                                <Link
                                    href={route('admin.doctors.index')}
                                    className="px-6 py-2.5 text-sm font-semibold text-gray-700 bg-white border border-gray-300 rounded-lg shadow-sm hover:bg-gray-50 transition-colors focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-teal-500"
                                >
                                    Cancel
                                </Link>
                                <PrimaryButton 
                                    disabled={processing} 
                                    className="px-6 py-2.5 bg-teal-600 hover:bg-teal-700 focus:ring-teal-500 shadow-md"
                                >
                                    <PlusCircleIcon className="w-5 h-5 mr-1.5" />
                                    {processing ? 'Adding Doctor...' : 'Add Doctor'}
                                </PrimaryButton>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}