import React, { useState } from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, useForm, Link } from '@inertiajs/react';
import PrimaryButton from '@/Components/PrimaryButton';
import InputLabel from '@/Components/InputLabel';
import TextInput from '@/Components/TextInput';
import InputError from '@/Components/InputError';
import { 
    ArrowLeftIcon, 
    PhotoIcon, 
    XMarkIcon,
    CalendarIcon,
    ClockIcon,
    MapPinIcon,
    CurrencyDollarIcon,
    TicketIcon,
    PlusCircleIcon // Icon for 'Create'
} from '@heroicons/react/24/outline';

// --- Reusable Component: FormSection (for clear grouping) ---
// Using the same clean, grid-based layout for professionalism
const FormSection = ({ title, description, children }) => (
    <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 lg:gap-8 border-t border-gray-200 pt-8">
        <div className="lg:col-span-1">
            <h3 className="text-xl font-bold text-gray-800 flex items-center">
                <PlusCircleIcon className="w-6 h-6 mr-2 text-emerald-500" />
                {title}
            </h3>
            <p className="mt-2 text-sm text-gray-500">{description}</p>
        </div>
        <div className="lg:col-span-2 space-y-6">
            {children}
        </div>
    </div>
);


export default function Create() {
    const { data, setData, post, processing, errors } = useForm({
        title: '',
        description: '',
        date: '',
        time: '',
        location: '',
        type: 'free',
        price: '',
        images: [],
    });

    const [imagePreviews, setImagePreviews] = useState([]);

    const handleImageChange = (e) => {
        const files = Array.from(e.target.files);
        setData('images', [...data.images, ...files]);

        files.forEach(file => {
            const reader = new FileReader();
            reader.onload = (e) => {
                setImagePreviews(prev => [...prev, e.target.result]);
            };
            reader.readAsDataURL(file);
        });
    };

    const handleRemoveImage = (index) => {
        const newImages = data.images.filter((_, i) => i !== index);
        const newImagePreviews = imagePreviews.filter((_, i) => i !== index);
        setData('images', newImages);
        setImagePreviews(newImagePreviews);
    };

    const submit = (e) => {
        e.preventDefault();
        post(route('admin.events.store'), {
            forceFormData: true,
        });
    };

    return (
        <AdminLayout>
            <Head title="Create New Event" />

            <div className="py-8">
                <div className="mx-auto sm:px-6 lg:px-8">
                    
                    {/* --- Page Header --- */}
                    <div className="mb-6 flex flex-col md:flex-row md:items-center md:justify-between">
                        <div className="flex-1 min-w-0">
                            <Link 
                                href={route('admin.events.index')} 
                                className="inline-flex items-center text-sm font-medium text-gray-500 hover:text-emerald-600 transition duration-150"
                            >
                                <ArrowLeftIcon className="-ml-0.5 mr-1 h-5 w-5" aria-hidden="true" />
                                Back to Events List
                            </Link>
                            <h1 className="mt-2 text-3xl font-extrabold leading-7 text-gray-900 sm:text-4xl sm:truncate">
                                **Publish a New Event** 🥳
                            </h1>
                        </div>
                    </div>

                    {/* --- Main Form Card --- */}
                    <div className="bg-white shadow-2xl sm:rounded-xl overflow-hidden border border-gray-100">
                        <form onSubmit={submit} className="p-6 sm:p-10 space-y-10">
                            
                            {/* Section 1: Event Details */}
                            <FormSection 
                                title="Basic Event Information"
                                description="Set the primary title and provide a detailed, engaging description for your event."
                            >
                                <div>
                                    <InputLabel htmlFor="title" value="Event Title" />
                                    <TextInput
                                        id="title"
                                        type="text"
                                        value={data.title}
                                        placeholder="E.g., Quarterly Marketing Strategy Session"
                                        className="mt-1 block w-full rounded-lg border-gray-300 focus:border-emerald-500 focus:ring-emerald-500 shadow-sm"
                                        onChange={(e) => setData('title', e.target.value)}
                                        required
                                        autoFocus
                                    />
                                    <InputError message={errors.title} className="mt-2" />
                                </div>
                                <div>
                                    <InputLabel htmlFor="description" value="Detailed Description" />
                                    <textarea
                                        id="description"
                                        value={data.description}
                                        rows="6"
                                        placeholder="Describe what attendees will experience or learn. Be clear and engaging!"
                                        className="mt-1 block w-full border-gray-300 focus:border-emerald-500 focus:ring-emerald-500 rounded-lg shadow-sm"
                                        onChange={(e) => setData('description', e.target.value)}
                                    ></textarea>
                                    <InputError message={errors.description} className="mt-2" />
                                </div>
                            </FormSection>

                            {/* Section 2: Date, Time & Location */}
                            <FormSection 
                                title="Scheduling & Venue"
                                description="Specify the exact date, time, and location to ensure attendees have accurate logistics."
                            >
                                <div className="grid grid-cols-1 sm:grid-cols-2 gap-6">
                                    {/* Date Field */}
                                    <div>
                                        <InputLabel htmlFor="date" value="Date" />
                                        <div className="mt-1 relative rounded-lg shadow-sm">
                                            <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                <CalendarIcon className="h-5 w-5 text-gray-400" aria-hidden="true" />
                                            </div>
                                            <TextInput
                                                id="date"
                                                type="date"
                                                value={data.date}
                                                className="mt-1 block w-full pl-10 rounded-lg border-gray-300 focus:border-emerald-500 focus:ring-emerald-500"
                                                onChange={(e) => setData('date', e.target.value)}
                                                required
                                            />
                                        </div>
                                        <InputError message={errors.date} className="mt-2" />
                                    </div>
                                    
                                    {/* Time Field */}
                                    <div>
                                        <InputLabel htmlFor="time" value="Time" />
                                        <div className="mt-1 relative rounded-lg shadow-sm">
                                            <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                <ClockIcon className="h-5 w-5 text-gray-400" aria-hidden="true" />
                                            </div>
                                            <TextInput
                                                id="time"
                                                type="time"
                                                value={data.time}
                                                className="mt-1 block w-full pl-10 rounded-lg border-gray-300 focus:border-emerald-500 focus:ring-emerald-500"
                                                onChange={(e) => setData('time', e.target.value)}
                                            />
                                        </div>
                                        <InputError message={errors.time} className="mt-2" />
                                    </div>
                                </div>
                                
                                {/* Location Field */}
                                <div>
                                    <InputLabel htmlFor="location" value="Location / Venue" />
                                    <div className="mt-1 relative rounded-lg shadow-sm">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <MapPinIcon className="h-5 w-5 text-gray-400" aria-hidden="true" />
                                        </div>
                                        <TextInput
                                            id="location"
                                            type="text"
                                            value={data.location}
                                            placeholder="e.g., Suite 101, Virtual Zoom Room, etc."
                                            className="mt-1 block w-full pl-10 rounded-lg border-gray-300 focus:border-emerald-500 focus:ring-emerald-500"
                                            onChange={(e) => setData('location', e.target.value)}
                                        />
                                    </div>
                                    <InputError message={errors.location} className="mt-2" />
                                </div>
                            </FormSection>

                            {/* Section 3: Pricing */}
                            <FormSection 
                                title="Ticketing & Price"
                                description="Define whether this is a free or paid event and set the corresponding price."
                            >
                                <div className="grid grid-cols-1 sm:grid-cols-2 gap-6">
                                    {/* Event Type Select */}
                                    <div>
                                        <InputLabel htmlFor="type" value="Event Type" />
                                        <div className="mt-1 relative rounded-lg shadow-sm">
                                            <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                <TicketIcon className="h-5 w-5 text-gray-400" aria-hidden="true" />
                                            </div>
                                            <select
                                                id="type"
                                                value={data.type}
                                                className="mt-1 block w-full pl-10 border-gray-300 focus:border-emerald-500 focus:ring-emerald-500 rounded-lg shadow-sm"
                                                onChange={(e) => setData('type', e.target.value)}
                                            >
                                                <option value="free">Free Event (No Ticket Required)</option>
                                                <option value="paid">Paid Ticket Required</option>
                                            </select>
                                        </div>
                                        <InputError message={errors.type} className="mt-2" />
                                    </div>
                                    
                                    {/* Price Field (Conditional) */}
                                    {data.type === 'paid' && (
                                        <div>
                                            <InputLabel htmlFor="price" value="Price (USD)" />
                                            <div className="mt-1 relative rounded-lg shadow-sm">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <CurrencyDollarIcon className="h-5 w-5 text-gray-400" aria-hidden="true" />
                                                </div>
                                                <TextInput
                                                    id="price"
                                                    type="number"
                                                    value={data.price}
                                                    placeholder="0.00"
                                                    className="mt-1 block w-full pl-10 rounded-lg border-gray-300 focus:border-emerald-500 focus:ring-emerald-500"
                                                    onChange={(e) => setData('price', e.target.value)}
                                                    step="0.01"
                                                />
                                            </div>
                                            <InputError message={errors.price} className="mt-2" />
                                        </div>
                                    )}
                                </div>
                            </FormSection>

                            {/* Section 4: Event Images */}
                            <FormSection 
                                title="Event Media & Branding"
                                description="Upload high-quality images to promote your event. (Max 10MB per file)"
                            >
                                {/* Drag & Drop Area */}
                                <div className="mt-2 flex justify-center px-6 pt-5 pb-6 border-2 border-emerald-300 border-dashed rounded-lg hover:border-emerald-400 transition-colors bg-emerald-50/50">
                                    <div className="space-y-1 text-center">
                                        <PhotoIcon className="mx-auto h-12 w-12 text-emerald-500" />
                                        <div className="flex text-sm text-gray-600">
                                            <label
                                                htmlFor="images"
                                                className="relative cursor-pointer bg-white rounded-md font-medium text-emerald-600 hover:text-emerald-700 focus-within:outline-none focus-within:ring-2 focus-within:ring-offset-2 focus-within:ring-emerald-500"
                                            >
                                                <span>Select files</span>
                                                <input id="images" name="images[]" type="file" multiple className="sr-only" onChange={handleImageChange} />
                                            </label>
                                            <p className="pl-1 text-gray-500">or drag and drop</p>
                                        </div>
                                        <p className="text-xs text-gray-500">PNG, JPG, GIF recommended.</p>
                                    </div>
                                </div>
                                <InputError message={errors.images} className="mt-2" />

                                {/* Image Previews Gallery */}
                                {imagePreviews.length > 0 && (
                                    <div className="mt-4 grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 gap-4">
                                        {imagePreviews.map((preview, index) => (
                                            <div key={index} className="relative group rounded-xl overflow-hidden border border-emerald-400 shadow-lg ring-2 ring-emerald-300 ring-offset-2">
                                                <img 
                                                    src={preview} 
                                                    alt={`Image Preview ${index + 1}`} 
                                                    className="h-28 w-full object-cover" 
                                                />
                                                <div className="absolute inset-0 bg-black/40 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center">
                                                    <button
                                                        type="button"
                                                        onClick={() => handleRemoveImage(index)}
                                                        className="bg-red-600 text-white rounded-full p-2 hover:bg-red-700 transition-colors shadow-xl"
                                                        title="Remove image"
                                                    >
                                                        <XMarkIcon className="h-5 w-5" />
                                                    </button>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                )}
                            </FormSection>


                            {/* --- Action Buttons --- */}
                            <div className="flex items-center justify-end space-x-4 pt-8 border-t border-gray-200">
                                <Link
                                    href={route('admin.events.index')}
                                    className="px-6 py-2.5 text-sm font-semibold text-gray-700 bg-white border border-gray-300 rounded-lg shadow-sm hover:bg-gray-50 transition-colors focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500"
                                >
                                    Cancel
                                </Link>
                                <PrimaryButton disabled={processing} className="px-6 py-2.5 bg-emerald-600 hover:bg-emerald-700 focus:ring-emerald-500 shadow-md">
                                    <PlusCircleIcon className="w-5 h-5 mr-1.5" />
                                    {processing ? 'Publishing...' : 'Publish Event'}
                                </PrimaryButton>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}