import React, { useState } from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link, usePage } from '@inertiajs/react';
import Pagination from '@/Components/Pagination';
import { FaCalendarAlt, FaMapMarkerAlt, FaSearch, FaFilter, FaEdit, FaTrash, FaEye, FaPlus, FaClock, FaUsers } from 'react-icons/fa';

export default function Index({ events }) {
    const { flash } = usePage().props;
    const [searchTerm, setSearchTerm] = useState('');
    const [filterType, setFilterType] = useState('all');

    // Filter events based on search and type
    const filteredEvents = events.data.filter(event => {
        const matchesSearch = event.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                             event.location.toLowerCase().includes(searchTerm.toLowerCase());
        const matchesType = filterType === 'all' || event.type === filterType;
        return matchesSearch && matchesType;
    });

    // Get event type color
    const getEventTypeColor = (type) => {
        switch(type) {
            case 'conference': return 'bg-blue-100 text-blue-800';
            case 'workshop': return 'bg-green-100 text-green-800';
            case 'seminar': return 'bg-purple-100 text-purple-800';
            case 'webinar': return 'bg-yellow-100 text-yellow-800';
            default: return 'bg-gray-100 text-gray-800';
        }
    };

    // Format date
    const formatDate = (dateString) => {
        const options = { year: 'numeric', month: 'short', day: 'numeric' };
        return new Date(dateString).toLocaleDateString(undefined, options);
    };

    // Check if event is upcoming or past
    const isEventUpcoming = (dateString) => {
        return new Date(dateString) > new Date();
    };

    const imageUrl = (imageString) => {
        let imageUrls = [];
        try {
            const parsedArray = JSON.parse(imageString);
            if (Array.isArray(parsedArray)) {
                imageUrls = parsedArray;
            }
            return imageUrls;
        } catch (e) {
            // Log an error if the string is malformed and cannot be parsed
            console.error("Error parsing image JSON string:", e);
        }
    }

    return (
        <AdminLayout>
            <Head title="Events Management" />

            <div className="py-6 px-4 sm:px-6 lg:px-8">
                {/* Page Header */}
                <div className="mb-8">
                    <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between">
                        <div>
                            <h1 className="text-3xl font-bold text-gray-900">Events Management</h1>
                            <p className="mt-1 text-sm text-gray-500">Manage and organize all your events in one place</p>
                        </div>
                        <div className="mt-4 sm:mt-0">
                            <Link
                                href={route('admin.events.create')}
                                className="inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-gradient-to-r from-green-600 to-emerald-600 hover:from-green-700 hover:to-emerald-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                            >
                                <FaPlus className="mr-2 -ml-1 h-4 w-4" />
                                Create Event
                            </Link>
                        </div>
                    </div>
                </div>

                {/* Flash Message */}
                {flash && flash.success && (
                    <div className="mb-6 p-4 bg-green-50 border-l-4 border-green-400 text-green-700 rounded-md shadow-sm">
                        <div className="flex">
                            <div className="flex-shrink-0">
                                <svg className="h-5 w-5 text-green-400" viewBox="0 0 20 20" fill="currentColor">
                                    <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                                </svg>
                            </div>
                            <div className="ml-3">
                                <p className="text-sm">{flash.success}</p>
                            </div>
                        </div>
                    </div>
                )}

                {/* Search and Filter Bar */}
                <div className="bg-white shadow rounded-lg p-4 mb-6">
                    <div className="flex flex-col sm:flex-row gap-4">
                        <div className="flex-1">
                            <div className="relative">
                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <FaSearch className="h-5 w-5 text-gray-400" />
                                </div>
                                <input
                                    type="text"
                                    className="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-md leading-5 bg-white placeholder-gray-500 focus:outline-none focus:placeholder-gray-400 focus:ring-1 focus:ring-green-500 focus:border-green-500 sm:text-sm"
                                    placeholder="Search events..."
                                    value={searchTerm}
                                    onChange={(e) => setSearchTerm(e.target.value)}
                                />
                            </div>
                        </div>
                        <div className="sm:w-48">
                            <div className="relative">
                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <FaFilter className="h-5 w-5 text-gray-400" />
                                </div>
                                <select
                                    className="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-md leading-5 bg-white focus:outline-none focus:ring-1 focus:ring-green-500 focus:border-green-500 sm:text-sm"
                                    value={filterType}
                                    onChange={(e) => setFilterType(e.target.value)}
                                >
                                    <option value="all">All Types</option>
                                    <option value="conference">Conference</option>
                                    <option value="workshop">Workshop</option>
                                    <option value="seminar">Seminar</option>
                                    <option value="webinar">Webinar</option>
                                </select>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Events Stats */}
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
                    <div className="bg-white overflow-hidden shadow rounded-lg">
                        <div className="p-5">
                            <div className="flex items-center">
                                <div className="flex-shrink-0 bg-green-500 rounded-md p-3">
                                    <FaCalendarAlt className="h-6 w-6 text-white" />
                                </div>
                                <div className="ml-5 w-0 flex-1">
                                    <dl>
                                        <dt className="text-sm font-medium text-gray-500 truncate">Total Events</dt>
                                        <dd className="text-lg font-medium text-gray-900">{events.data.length}</dd>
                                    </dl>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div className="bg-white overflow-hidden shadow rounded-lg">
                        <div className="p-5">
                            <div className="flex items-center">
                                <div className="flex-shrink-0 bg-blue-500 rounded-md p-3">
                                    <FaClock className="h-6 w-6 text-white" />
                                </div>
                                <div className="ml-5 w-0 flex-1">
                                    <dl>
                                        <dt className="text-sm font-medium text-gray-500 truncate">Upcoming</dt>
                                        <dd className="text-lg font-medium text-gray-900">
                                            {events.data.filter(event => isEventUpcoming(event.date)).length}
                                        </dd>
                                    </dl>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div className="bg-white overflow-hidden shadow rounded-lg">
                        <div className="p-5">
                            <div className="flex items-center">
                                <div className="flex-shrink-0 bg-purple-500 rounded-md p-3">
                                    <FaUsers className="h-6 w-6 text-white" />
                                </div>
                                <div className="ml-5 w-0 flex-1">
                                    <dl>
                                        <dt className="text-sm font-medium text-gray-500 truncate">Total Attendees</dt>
                                        <dd className="text-lg font-medium text-gray-900">0</dd>
                                    </dl>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div className="bg-white overflow-hidden shadow rounded-lg">
                        <div className="p-5">
                            <div className="flex items-center">
                                <div className="flex-shrink-0 bg-yellow-500 rounded-md p-3">
                                    <FaMapMarkerAlt className="h-6 w-6 text-white" />
                                </div>
                                <div className="ml-5 w-0 flex-1">
                                    <dl>
                                        <dt className="text-sm font-medium text-gray-500 truncate">Locations</dt>
                                        <dd className="text-lg font-medium text-gray-900">0</dd>
                                    </dl>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Events Table */}
                <div className="bg-white shadow overflow-hidden sm:rounded-md">
                    <ul className="divide-y divide-gray-200">
                        {filteredEvents.length > 0 ? (
                            filteredEvents.map((event) => (
                                <li key={event.id}>
                                    <div className="block hover:bg-gray-50">
                                        <div className="px-4 py-4 sm:px-6">
                                            <div className="flex items-center justify-between">
                                                <div className="flex items-center">
                                                    <div className="flex-shrink-0">
                                                        {event.images && event.images.length > 0 ? (
                                                            <img className="h-12 w-12 rounded-lg object-cover" src={imageUrl(event.images)[0]} alt={event.title} />
                                                        ) : (
                                                            <div className="h-12 w-12 rounded-lg bg-gray-200 flex items-center justify-center">
                                                                <FaCalendarAlt className="h-6 w-6 text-gray-400" />
                                                            </div>
                                                        )}
                                                    </div>
                                                    <div className="ml-4">
                                                        <div className="flex items-center">
                                                            <p className="text-sm font-medium text-gray-900 truncate">{event.title}</p>
                                                            <span className={`ml-2 inline-flex px-2 py-0.5 rounded-full text-xs font-medium ${getEventTypeColor(event.type)}`}>
                                                                {event.type}
                                                            </span>
                                                            {isEventUpcoming(event.date) && (
                                                                <span className="ml-2 inline-flex px-2 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                                                    Upcoming
                                                                </span>
                                                            )}
                                                        </div>
                                                        <div className="mt-1 flex items-center text-sm text-gray-500">
                                                            <FaCalendarAlt className="flex-shrink-0 mr-1.5 h-4 w-4 text-gray-400" />
                                                            <p>{formatDate(event.date)}</p>
                                                            <FaMapMarkerAlt className="flex-shrink-0 ml-4 mr-1.5 h-4 w-4 text-gray-400" />
                                                            <p>{event.location}</p>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div className="flex items-center space-x-2">
                                                    <Link
                                                        href={route('admin.events.show', event.id)}
                                                        className="inline-flex items-center px-3 py-1.5 border border-gray-300 shadow-sm text-xs font-medium rounded text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                                    >
                                                        <FaEye className="mr-1 h-3 w-3" />
                                                        View
                                                    </Link>
                                                    <Link
                                                        href={route('admin.events.edit', event.id)}
                                                        className="inline-flex items-center px-3 py-1.5 border border-transparent shadow-sm text-xs font-medium rounded text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                                    >
                                                        <FaEdit className="mr-1 h-3 w-3" />
                                                        Edit
                                                    </Link>
                                                    <Link
                                                        href={route('admin.events.destroy', event.id)}
                                                        method="delete"
                                                        as="button"
                                                        className="inline-flex items-center px-3 py-1.5 border border-transparent shadow-sm text-xs font-medium rounded text-white bg-red-600 hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500"
                                                    >
                                                        <FaTrash className="mr-1 h-3 w-3" />
                                                        Delete
                                                    </Link>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </li>
                            ))
                        ) : (
                            <li className="px-4 py-8 text-center">
                                <div className="text-gray-500">
                                    <FaCalendarAlt className="mx-auto h-12 w-12 text-gray-400" />
                                    <h3 className="mt-2 text-sm font-medium text-gray-900">No events found</h3>
                                    <p className="mt-1 text-sm text-gray-500">Get started by creating a new event.</p>
                                    <div className="mt-6">
                                        <Link
                                            href={route('admin.events.create')}
                                            className="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                        >
                                            <FaPlus className="mr-2 -ml-1 h-4 w-4" />
                                            Create Event
                                        </Link>
                                    </div>
                                </div>
                            </li>
                        )}
                    </ul>
                </div>

                {/* Pagination */}
                {events.links && (
                    <div className="mt-6">
                        <Pagination links={events.links} />
                    </div>
                )}
            </div>
        </AdminLayout>
    );
}