import React, { useState } from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, useForm } from '@inertiajs/react';
import { usePage } from '@inertiajs/react';
import { showErrorToast, showSuccessToast } from '@/Components/ToastNote';

export default function Edit({ paymentGateways }) {
    const { flash } = usePage().props;
    const [activeTab, setActiveTab] = useState('paystack');
    const [showSecrets, setShowSecrets] = useState({
        paystack_secret_key: false,
        paypal_sandbox_client_secret: false,
        paypal_live_client_secret: false
    });
    const [enabledGateways, setEnabledGateways] = useState({
        paystack: paymentGateways?.paystack?.enabled || false,
        paypal: paymentGateways?.paypal?.enabled || false
    });

    const { data, setData, put, processing, errors, reset } = useForm({
        paystack_public_key: paymentGateways?.paystack?.public_key || '',
        paystack_secret_key: paymentGateways?.paystack?.secret_key || '',
        paystack_merchant_email: paymentGateways?.paystack?.merchant_email || '',
        paystack_payment_url: paymentGateways?.paystack?.payment_url || 'https://api.paystack.co',
        paystack_enabled: paymentGateways?.paystack?.enabled || false,
        paypal_mode: paymentGateways?.paypal?.mode || 'sandbox',
        paypal_sandbox_client_id: paymentGateways?.paypal?.sandbox_client_id || '',
        paypal_sandbox_client_secret: paymentGateways?.paypal?.sandbox_client_secret || '',
        paypal_live_client_id: paymentGateways?.paypal?.live_client_id || '',
        paypal_live_client_secret: paymentGateways?.paypal?.live_client_secret || '',
        paypal_currency: paymentGateways?.paypal?.currency || 'USD',
        paypal_enabled: paymentGateways?.paypal?.enabled || false,
    });

    const submit = (e) => {
        e.preventDefault();
        put(route('admin.payment-settings.update'), {
            onSuccess: () => {
                showSuccessToast('Payment settings updated successfully!');
            },
            onError: (errors) => {
                showErrorToast('Failed to update payment settings. Please check the form for errors.');
            }
        });
    };

    const toggleSecretVisibility = (field) => {
        setShowSecrets(prev => ({
            ...prev,
            [field]: !prev[field]
        }));
    };

    const toggleGateway = (gateway) => {
        setEnabledGateways(prev => ({
            ...prev,
            [gateway]: !prev[gateway]
        }));
        setData(`${gateway}_enabled`, !enabledGateways[gateway]);
    };

    const tabs = [
        { id: 'paystack', name: 'PayStack', icon: 'M5 12h14M5 12a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v4a2 2 0 01-2 2M5 12a2 2 0 00-2 2v4a2 2 0 002 2h14a2 2 0 002-2v-4a2 2 0 00-2-2m-2-4h.01M17 16h.01' },
        { id: 'paypal', name: 'PayPal', icon: 'M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z' }
    ];

    return (
        <AdminLayout
            header={
                <div className="flex items-center justify-between">
                    <h2 className="text-xl font-semibold leading-tight text-gray-800">Payment Settings</h2>
                    <div className="flex items-center space-x-2 text-sm text-gray-600">
                        <span>Dashboard</span>
                        <span>/</span>
                        <span>Payment Settings</span>
                    </div>
                </div>
            }
        >
            <Head title="Payment Settings" />

            <div className="py-8">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8 space-y-6">
                    {/* Page Header */}
                    <div className="bg-gradient-to-r from-green-600 to-emerald-700 rounded-xl p-6 text-white shadow-lg">
                        <h1 className="text-2xl font-bold">Payment Gateway Configuration</h1>
                        <p className="mt-2 opacity-90">Manage your payment gateway settings and credentials</p>
                    </div>

                    {/* Flash Messages */}
                    {flash && flash.success && (
                        <div className="bg-green-50 border-l-4 border-green-400 p-4 rounded-md">
                            <div className="flex">
                                <div className="flex-shrink-0">
                                    <svg className="h-5 w-5 text-green-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                        <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                                    </svg>
                                </div>
                                <div className="ml-3">
                                    <p className="text-sm text-green-700">{flash.success}</p>
                                </div>
                            </div>
                        </div>
                    )}
                    {flash && flash.error && (
                        <div className="bg-red-50 border-l-4 border-red-400 p-4 rounded-md">
                            <div className="flex">
                                <div className="flex-shrink-0">
                                    <svg className="h-5 w-5 text-red-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                        <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
                                    </svg>
                                </div>
                                <div className="ml-3">
                                    <p className="text-sm text-red-700">{flash.error}</p>
                                </div>
                            </div>
                        </div>
                    )}

                    {/* Payment Gateway Status Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        {tabs.map((tab) => (
                            <div key={tab.id} className="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
                                <div className="p-6">
                                    <div className="flex items-center justify-between mb-4">
                                        <div className="flex items-center">
                                            <div className={`p-2 rounded-lg ${enabledGateways[tab.id] ? 'bg-green-100' : 'bg-gray-100'}`}>
                                                <svg className={`h-6 w-6 ${enabledGateways[tab.id] ? 'text-green-600' : 'text-gray-400'}`} fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d={tab.icon} />
                                                </svg>
                                            </div>
                                            <div className="ml-4">
                                                <h3 className="text-lg font-medium text-gray-900">{tab.name}</h3>
                                                <p className="text-sm text-gray-500">
                                                    {enabledGateways[tab.id] ? 'Active' : 'Inactive'}
                                                </p>
                                            </div>
                                        </div>
                                        <button
                                            type="button"
                                            onClick={() => toggleGateway(tab.id)}
                                            className={`relative inline-flex flex-shrink-0 h-6 w-11 border-2 border-transparent rounded-full cursor-pointer transition-colors ease-in-out duration-200 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 ${
                                                enabledGateways[tab.id] ? 'bg-green-600' : 'bg-gray-200'
                                            }`}
                                        >
                                            <span className="translate-x-0 inline-block h-5 w-5 rounded-full bg-white shadow transform ring-0 transition ease-in-out duration-200"
                                                style={{ transform: enabledGateways[tab.id] ? 'translateX(20px)' : 'translateX(0)' }}
                                            />
                                        </button>
                                    </div>
                                    <div className="text-sm text-gray-600">
                                        {tab.id === 'paystack' ? (
                                            <p>PayStack allows you to accept payments from customers across Africa.</p>
                                        ) : (
                                            <p>PayPal is a global payment solution that allows you to accept payments worldwide.</p>
                                        )}
                                    </div>
                                </div>
                            </div>
                        ))}
                    </div>

                    {/* Configuration Form */}
                    <div className="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
                        {/* Tab Navigation */}
                        <div className="border-b border-gray-200">
                            <nav className="flex -mb-px">
                                {tabs.map((tab) => (
                                    <button
                                        key={tab.id}
                                        type="button"
                                        onClick={() => setActiveTab(tab.id)}
                                        className={`py-4 px-6 text-sm font-medium border-b-2 transition-colors duration-200 ${
                                            activeTab === tab.id
                                                ? 'border-green-500 text-green-600'
                                                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                        }`}
                                    >
                                        <div className="flex items-center">
                                            <svg className="h-5 w-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d={tab.icon} />
                                            </svg>
                                            {tab.name}
                                        </div>
                                    </button>
                                ))}
                            </nav>
                        </div>

                        <form onSubmit={submit}>
                            {/* PayStack Settings */}
                            {activeTab === 'paystack' && (
                                <div className="p-6 space-y-6">
                                    <div className="bg-blue-50 border-l-4 border-blue-400 p-4 rounded-md">
                                        <div className="flex">
                                            <div className="flex-shrink-0">
                                                <svg className="h-5 w-5 text-blue-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                    <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
                                                </svg>
                                            </div>
                                            <div className="ml-3">
                                                <p className="text-sm text-blue-700">
                                                    You can find your PayStack API keys in your PayStack dashboard under Settings &gt; API Keys.
                                                </p>
                                            </div>
                                        </div>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label htmlFor="paystack_public_key" className="block text-sm font-medium text-gray-700 mb-1">Public Key</label>
                                            <div className="relative">
                                                <input
                                                    type="text"
                                                    id="paystack_public_key"
                                                    value={data.paystack_public_key}
                                                    onChange={(e) => setData('paystack_public_key', e.target.value)}
                                                    className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.paystack_public_key ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="pk_test_xxxxxxxxxxxxxxxxxxxxxxxx"
                                                />
                                                {errors.paystack_public_key && (
                                                    <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                                                        <svg className="h-5 w-5 text-red-500" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                            <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                        </svg>
                                                    </div>
                                                )}
                                            </div>
                                            {errors.paystack_public_key && (
                                                <p className="mt-1 text-sm text-red-600">{errors.paystack_public_key}</p>
                                            )}
                                        </div>

                                        <div>
                                            <label htmlFor="paystack_merchant_email" className="block text-sm font-medium text-gray-700 mb-1">Merchant Email</label>
                                            <div className="relative">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <svg className="h-5 w-5 text-gray-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                        <path d="M2.003 5.884L10 9.882l7.997-3.998A2 2 0 0016 4H4a2 2 0 00-1.997 1.884z" />
                                                        <path d="M18 8.118l-8 4-8-4V14a2 2 0 002 2h12a2 2 0 002-2V8.118z" />
                                                    </svg>
                                                </div>
                                                <input
                                                    type="email"
                                                    id="paystack_merchant_email"
                                                    value={data.paystack_merchant_email}
                                                    onChange={(e) => setData('paystack_merchant_email', e.target.value)}
                                                    className={`mt-1 block w-full pl-10 rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.paystack_merchant_email ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="merchant@example.com"
                                                />
                                                {errors.paystack_merchant_email && (
                                                    <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                                                        <svg className="h-5 w-5 text-red-500" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                            <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                        </svg>
                                                    </div>
                                                )}
                                            </div>
                                            {errors.paystack_merchant_email && (
                                                <p className="mt-1 text-sm text-red-600">{errors.paystack_merchant_email}</p>
                                            )}
                                        </div>

                                        <div>
                                            <label htmlFor="paystack_payment_url" className="block text-sm font-medium text-gray-700 mb-1">Payment URL</label>
                                            <div className="relative">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <svg className="h-5 w-5 text-gray-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                        <path fillRule="evenodd" d="M12.586 4.586a2 2 0 112.828 2.828l-3 3a2 2 0 01-2.828 0 1 1 0 00-1.414 1.414 4 4 0 005.656 0l3-3a4 4 0 00-5.656-5.656l-1.5 1.5a1 1 0 101.414 1.414l1.5-1.5zm-5 5a2 2 0 012.828 0 1 1 0 101.414-1.414 2 2 0 00-2.828 0l-3 3a2 2 0 000 2.828 1 1 0 101.414-1.414l3-3z" clipRule="evenodd" />
                                                    </svg>
                                                </div>
                                                <input
                                                    type="url"
                                                    id="paystack_payment_url"
                                                    value={data.paystack_payment_url}
                                                    onChange={(e) => setData('paystack_payment_url', e.target.value)}
                                                    className={`mt-1 block w-full pl-10 rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.paystack_payment_url ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="https://api.paystack.co"
                                                />
                                                {errors.paystack_payment_url && (
                                                    <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                                                        <svg className="h-5 w-5 text-red-500" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                            <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                        </svg>
                                                    </div>
                                                )}
                                            </div>
                                            {errors.paystack_payment_url && (
                                                <p className="mt-1 text-sm text-red-600">{errors.paystack_payment_url}</p>
                                            )}
                                        </div>

                                        <div className="md:col-span-2">
                                            <label htmlFor="paystack_secret_key" className="block text-sm font-medium text-gray-700 mb-1">Secret Key</label>
                                            <div className="relative">
                                                <input
                                                    type={showSecrets.paystack_secret_key ? "text" : "password"}
                                                    id="paystack_secret_key"
                                                    value={data.paystack_secret_key}
                                                    onChange={(e) => setData('paystack_secret_key', e.target.value)}
                                                    className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.paystack_secret_key ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="sk_test_xxxxxxxxxxxxxxxxxxxxxxxx"
                                                />
                                                <div className="absolute inset-y-0 right-0 flex items-center pr-3">
                                                    <button
                                                        type="button"
                                                        onClick={() => toggleSecretVisibility('paystack_secret_key')}
                                                        className="text-gray-400 hover:text-gray-500 focus:outline-none"
                                                    >
                                                        {showSecrets.paystack_secret_key ? (
                                                            <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                                <path d="M10 12a2 2 0 100-4 2 2 0 000 4z" />
                                                                <path fillRule="evenodd" d="M.458 10C1.732 5.943 5.522 3 10 3s8.268 2.943 9.542 7c-1.274 4.057-5.064 7-9.542 7S1.732 14.057.458 10zM14 10a4 4 0 11-8 0 4 4 0 018 0z" clipRule="evenodd" />
                                                            </svg>
                                                        ) : (
                                                            <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                                <path fillRule="evenodd" d="M3.707 2.293a1 1 0 00-1.414 1.414l14 14a1 1 0 001.414-1.414l-1.473-1.473A10.014 10.014 0 0019.542 10C18.268 5.943 14.478 3 10 3a9.958 9.958 0 00-4.512 1.074l-1.78-1.781zm4.261 4.26l1.514 1.515a2.003 2.003 0 012.45 2.45l1.514 1.514a4 4 0 00-5.478-5.478z" clipRule="evenodd" />
                                                                <path d="M12.454 16.697L9.75 13.992a4 4 0 01-3.742-3.741L2.335 6.578A9.98 9.98 0 00.458 10c1.274 4.057 5.065 7 9.542 7 .847 0 1.669-.105 2.454-.303z" />
                                                            </svg>
                                                        )}
                                                    </button>
                                                </div>
                                            </div>
                                            {errors.paystack_secret_key && (
                                                <p className="mt-1 text-sm text-red-600">{errors.paystack_secret_key}</p>
                                            )}
                                        </div>
                                    </div>
                                </div>
                            )}

                            {/* PayPal Settings */}
                            {activeTab === 'paypal' && (
                                <div className="p-6 space-y-6">
                                    <div className="bg-blue-50 border-l-4 border-blue-400 p-4 rounded-md">
                                        <div className="flex">
                                            <div className="flex-shrink-0">
                                                <svg className="h-5 w-5 text-blue-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                    <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
                                                </svg>
                                            </div>
                                            <div className="ml-3">
                                                <p className="text-sm text-blue-700">
                                                    You can find your PayPal API credentials in your PayPal developer dashboard under My Apps & Credentials.
                                                </p>
                                            </div>
                                        </div>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label htmlFor="paypal_mode" className="block text-sm font-medium text-gray-700 mb-1">Mode</label>
                                            <select
                                                id="paypal_mode"
                                                value={data.paypal_mode}
                                                onChange={(e) => setData('paypal_mode', e.target.value)}
                                                className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                    errors.paypal_mode ? 'border-red-300' : 'border-gray-300'
                                                }`}
                                            >
                                                <option value="sandbox">Sandbox (Testing)</option>
                                                <option value="live">Live (Production)</option>
                                            </select>
                                            {errors.paypal_mode && (
                                                <p className="mt-1 text-sm text-red-600">{errors.paypal_mode}</p>
                                            )}
                                        </div>

                                        <div>
                                            <label htmlFor="paypal_currency" className="block text-sm font-medium text-gray-700 mb-1">Currency</label>
                                            <select
                                                id="paypal_currency"
                                                value={data.paypal_currency}
                                                onChange={(e) => setData('paypal_currency', e.target.value)}
                                                className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                    errors.paypal_currency ? 'border-red-300' : 'border-gray-300'
                                                }`}
                                            >
                                                <option value="USD">USD - US Dollar</option>
                                                <option value="EUR">EUR - Euro</option>
                                                <option value="GBP">GBP - British Pound</option>
                                                <option value="CAD">CAD - Canadian Dollar</option>
                                                <option value="AUD">AUD - Australian Dollar</option>
                                                <option value="JPY">JPY - Japanese Yen</option>
                                                <option value="CNY">CNY - Chinese Yuan</option>
                                                <option value="INR">INR - Indian Rupee</option>
                                            </select>
                                            {errors.paypal_currency && (
                                                <p className="mt-1 text-sm text-red-600">{errors.paypal_currency}</p>
                                            )}
                                        </div>

                                        {data.paypal_mode === 'sandbox' ? (
                                            <>
                                                <div>
                                                    <label htmlFor="paypal_sandbox_client_id" className="block text-sm font-medium text-gray-700 mb-1">Sandbox Client ID</label>
                                                    <div className="relative">
                                                        <input
                                                            type="text"
                                                            id="paypal_sandbox_client_id"
                                                            value={data.paypal_sandbox_client_id}
                                                            onChange={(e) => setData('paypal_sandbox_client_id', e.target.value)}
                                                            className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                                errors.paypal_sandbox_client_id ? 'border-red-300' : 'border-gray-300'
                                                            }`}
                                                            placeholder="AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQqRrSsTtUuVvWwXxYyZz"
                                                        />
                                                        {errors.paypal_sandbox_client_id && (
                                                            <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                                                                <svg className="h-5 w-5 text-red-500" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                                    <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                                </svg>
                                                            </div>
                                                        )}
                                                    </div>
                                                    {errors.paypal_sandbox_client_id && (
                                                        <p className="mt-1 text-sm text-red-600">{errors.paypal_sandbox_client_id}</p>
                                                    )}
                                                </div>

                                                <div>
                                                    <label htmlFor="paypal_sandbox_client_secret" className="block text-sm font-medium text-gray-700 mb-1">Sandbox Client Secret</label>
                                                    <div className="relative">
                                                        <input
                                                            type={showSecrets.paypal_sandbox_client_secret ? "text" : "password"}
                                                            id="paypal_sandbox_client_secret"
                                                            value={data.paypal_sandbox_client_secret}
                                                            onChange={(e) => setData('paypal_sandbox_client_secret', e.target.value)}
                                                            className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                                errors.paypal_sandbox_client_secret ? 'border-red-300' : 'border-gray-300'
                                                            }`}
                                                            placeholder="AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQqRrSsTtUuVvWwXxYyZz"
                                                        />
                                                        <div className="absolute inset-y-0 right-0 flex items-center pr-3">
                                                            <button
                                                                type="button"
                                                                onClick={() => toggleSecretVisibility('paypal_sandbox_client_secret')}
                                                                className="text-gray-400 hover:text-gray-500 focus:outline-none"
                                                            >
                                                                {showSecrets.paypal_sandbox_client_secret ? (
                                                                    <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                                        <path d="M10 12a2 2 0 100-4 2 2 0 000 4z" />
                                                                        <path fillRule="evenodd" d="M.458 10C1.732 5.943 5.522 3 10 3s8.268 2.943 9.542 7c-1.274 4.057-5.064 7-9.542 7S1.732 14.057.458 10zM14 10a4 4 0 11-8 0 4 4 0 018 0z" clipRule="evenodd" />
                                                                    </svg>
                                                                ) : (
                                                                    <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                                        <path fillRule="evenodd" d="M3.707 2.293a1 1 0 00-1.414 1.414l14 14a1 1 0 001.414-1.414l-1.473-1.473A10.014 10.014 0 0019.542 10C18.268 5.943 14.478 3 10 3a9.958 9.958 0 00-4.512 1.074l-1.78-1.781zm4.261 4.26l1.514 1.515a2.003 2.003 0 012.45 2.45l1.514 1.514a4 4 0 00-5.478-5.478z" clipRule="evenodd" />
                                                                        <path d="M12.454 16.697L9.75 13.992a4 4 0 01-3.742-3.741L2.335 6.578A9.98 9.98 0 00.458 10c1.274 4.057 5.065 7 9.542 7 .847 0 1.669-.105 2.454-.303z" />
                                                                    </svg>
                                                                )}
                                                            </button>
                                                        </div>
                                                    </div>
                                                    {errors.paypal_sandbox_client_secret && (
                                                        <p className="mt-1 text-sm text-red-600">{errors.paypal_sandbox_client_secret}</p>
                                                    )}
                                                </div>
                                            </>
                                        ) : (
                                            <>
                                                <div>
                                                    <label htmlFor="paypal_live_client_id" className="block text-sm font-medium text-gray-700 mb-1">Live Client ID</label>
                                                    <div className="relative">
                                                        <input
                                                            type="text"
                                                            id="paypal_live_client_id"
                                                            value={data.paypal_live_client_id}
                                                            onChange={(e) => setData('paypal_live_client_id', e.target.value)}
                                                            className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                                errors.paypal_live_client_id ? 'border-red-300' : 'border-gray-300'
                                                            }`}
                                                            placeholder="AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQqRrSsTtUuVvWwXxYyZz"
                                                        />
                                                        {errors.paypal_live_client_id && (
                                                            <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                                                                <svg className="h-5 w-5 text-red-500" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                                    <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                                </svg>
                                                            </div>
                                                        )}
                                                    </div>
                                                    {errors.paypal_live_client_id && (
                                                        <p className="mt-1 text-sm text-red-600">{errors.paypal_live_client_id}</p>
                                                    )}
                                                </div>

                                                <div>
                                                    <label htmlFor="paypal_live_client_secret" className="block text-sm font-medium text-gray-700 mb-1">Live Client Secret</label>
                                                    <div className="relative">
                                                        <input
                                                            type={showSecrets.paypal_live_client_secret ? "text" : "password"}
                                                            id="paypal_live_client_secret"
                                                            value={data.paypal_live_client_secret}
                                                            onChange={(e) => setData('paypal_live_client_secret', e.target.value)}
                                                            className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                                errors.paypal_live_client_secret ? 'border-red-300' : 'border-gray-300'
                                                            }`}
                                                            placeholder="AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQqRrSsTtUuVvWwXxYyZz"
                                                        />
                                                        <div className="absolute inset-y-0 right-0 flex items-center pr-3">
                                                            <button
                                                                type="button"
                                                                onClick={() => toggleSecretVisibility('paypal_live_client_secret')}
                                                                className="text-gray-400 hover:text-gray-500 focus:outline-none"
                                                            >
                                                                {showSecrets.paypal_live_client_secret ? (
                                                                    <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                                        <path d="M10 12a2 2 0 100-4 2 2 0 000 4z" />
                                                                        <path fillRule="evenodd" d="M.458 10C1.732 5.943 5.522 3 10 3s8.268 2.943 9.542 7c-1.274 4.057-5.064 7-9.542 7S1.732 14.057.458 10zM14 10a4 4 0 11-8 0 4 4 0 018 0z" clipRule="evenodd" />
                                                                    </svg>
                                                                ) : (
                                                                    <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                                        <path fillRule="evenodd" d="M3.707 2.293a1 1 0 00-1.414 1.414l14 14a1 1 0 001.414-1.414l-1.473-1.473A10.014 10.014 0 0019.542 10C18.268 5.943 14.478 3 10 3a9.958 9.958 0 00-4.512 1.074l-1.78-1.781zm4.261 4.26l1.514 1.515a2.003 2.003 0 012.45 2.45l1.514 1.514a4 4 0 00-5.478-5.478z" clipRule="evenodd" />
                                                                        <path d="M12.454 16.697L9.75 13.992a4 4 0 01-3.742-3.741L2.335 6.578A9.98 9.98 0 00.458 10c1.274 4.057 5.065 7 9.542 7 .847 0 1.669-.105 2.454-.303z" />
                                                                    </svg>
                                                                )}
                                                            </button>
                                                        </div>
                                                    </div>
                                                    {errors.paypal_live_client_secret && (
                                                        <p className="mt-1 text-sm text-red-600">{errors.paypal_live_client_secret}</p>
                                                    )}
                                                </div>
                                            </>
                                        )}
                                    </div>
                                </div>
                            )}

                            {/* Action Buttons */}
                            <div className="px-6 py-4 bg-gray-50 border-t border-gray-200 flex justify-end space-x-3">
                                <button
                                    type="button"
                                    onClick={() => reset()}
                                    className="px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                >
                                    Cancel
                                </button>
                                <button
                                    type="submit"
                                    className="px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 disabled:opacity-50 transition-all duration-200 flex items-center"
                                    disabled={processing}
                                >
                                    {processing ? (
                                        <>
                                            <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                                <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                                <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                            </svg>
                                            Saving...
                                        </>
                                    ) : (
                                        <>
                                            <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                                            </svg>
                                            Save Settings
                                        </>
                                    )}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}