import React from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, useForm, Link } from '@inertiajs/react';
import InputLabel from '@/Components/InputLabel';
import TextInput from '@/Components/TextInput';
import InputError from '@/Components/InputError';
import PrimaryButton from '@/Components/PrimaryButton';
import SecondaryButton from '@/Components/SecondaryButton';
import { showErrorToast, showSuccessToast } from '@/Components/ToastNote';
import { TagIcon, DocumentTextIcon, ArrowLeftIcon } from '@heroicons/react/24/outline';

export default function Create({ auth }) {
    const { data, setData, post, processing, errors, reset } = useForm({
        name: '',
        description: '',
    });

    const submit = (e) => {
        e.preventDefault();
        post(route('admin.product-categories.store'), {
            onSuccess: () => {
                showSuccessToast('Product category created successfully!');
                reset();
            },
            onError: () => {
                showErrorToast('Failed to create product category. Please check the form for errors.');
            },
        });
    };

    return (
        <AdminLayout user={auth.user}>
            <Head title="Create Product Category" />

            <div className="py-6">
                <div className="max-w-2xl mx-auto sm:px-6 lg:px-8">
                    {/* --- Header Section --- */}
                    <header className="mb-6">
                        <Link 
                            href={route('admin.product-categories.index')} 
                            className="text-sm font-medium text-emerald-600 hover:text-emerald-500 flex items-center mb-2 transition-colors"
                        >
                            <ArrowLeftIcon className="w-4 h-4 mr-1" />
                            Back to Categories
                        </Link>
                        <h1 className="text-3xl font-extrabold text-gray-900">Create New Category</h1>
                        <p className="mt-1 text-base text-gray-500">
                            Add a new category to organize your products. A clear name and description help customers find what they're looking for.
                        </p>
                    </header>
                    
                    {/* --- Form Card --- */}
                    <div className="bg-white shadow-xl rounded-lg overflow-hidden">
                        <form onSubmit={submit} className="p-8 space-y-8">
                            
                            {/* --- Category Name Field --- */}
                            <div>
                                <InputLabel htmlFor="name" value="Category Name" />
                                <div className="mt-1 relative rounded-md shadow-sm">
                                    <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                        <TagIcon className="h-5 w-5 text-gray-400" aria-hidden="true" />
                                    </div>
                                    <TextInput
                                        id="name"
                                        type="text"
                                        name="name"
                                        value={data.name}
                                        className="block w-full pl-10"
                                        placeholder="e.g., Women's Clothing"
                                        isFocused={true}
                                        onChange={(e) => setData('name', e.target.value)}
                                        required
                                    />
                                </div>
                                <p className="mt-2 text-sm text-gray-500">
                                    Choose a clear and concise name. This will be displayed to customers and used for sorting.
                                </p>
                                <InputError message={errors.name} className="mt-2" />
                            </div>

                            {/* --- Description Field --- */}
                            <div>
                                <InputLabel htmlFor="description" value="Description" />
                                <div className="mt-1 relative rounded-md shadow-sm">
                                    <div className="absolute top-3 left-3 pl-3 flex items-start pointer-events-none">
                                        <DocumentTextIcon className="h-5 w-5 text-gray-400" aria-hidden="true" />
                                    </div>
                                    <textarea
                                        id="description"
                                        name="description"
                                        value={data.description}
                                        rows="4"
                                        className="block w-full pl-10 border-gray-300 focus:border-emerald-500 focus:ring-emerald-500 rounded-md"
                                        placeholder="e.g., Explore our latest collection of comfortable and stylish apparel for women, including dresses, tops, and more."
                                        onChange={(e) => setData('description', e.target.value)}
                                    ></textarea>
                                </div>
                                <p className="mt-2 text-sm text-gray-500">
                                    Provide a brief overview of the types of products in this category. This helps with search and improves the user experience.
                                </p>
                                <InputError message={errors.description} className="mt-2" />
                            </div>

                            {/* --- Action Buttons --- */}
                            <div className="flex items-center justify-end space-x-4 pt-6 border-t border-gray-200">
                                <Link href={route('admin.product-categories.index')}>
                                    <SecondaryButton type="button">Cancel</SecondaryButton>
                                </Link>
                                <PrimaryButton disabled={processing}>
                                    {processing ? 'Saving...' : 'Save Category'}
                                </PrimaryButton>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}