import React, { useState } from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link } from '@inertiajs/react';
import { 
    ChevronLeftIcon, 
    ChevronRightIcon, 
    PencilIcon, 
    TrashIcon, 
    MagnifyingGlassIcon,
    CubeIcon, 
    PlusIcon,
    Bars3CenterLeftIcon, 
    ExclamationTriangleIcon, 
    ShieldCheckIcon, 
    ChartBarIcon 
} from '@heroicons/react/24/outline';

// --- CURRENCY CONSTANT ---
const CURRENCY_SYMBOL = 'GH₵';

// --- Helper function to calculate product analytics ---
const getProductAnalytics = (products) => {
    const totalProducts = products.length;
    let totalInventoryValue = 0;
    let lowStockCount = 0;
    let outOfStockCount = 0;

    products.forEach(product => {
        const stock = parseFloat(product.stock || 0);
        const price = parseFloat(product.price || 0);
        
        // Calculate total inventory value
        totalInventoryValue += stock * price;

        // Check stock status for physical products only (assuming is_digital is false/null)
        if (!product.is_digital) {
            if (stock > 0 && stock <= 10) {
                lowStockCount++;
            } else if (stock === 0) {
                outOfStockCount++;
            }
        }
    });

    return { 
        totalProducts, 
        totalInventoryValue: totalInventoryValue.toFixed(2), 
        lowStockCount, 
        outOfStockCount 
    };
};

// --- Sub-Component: Analytics Card (UPDATED CLASS: text-3xl font-semibold) ---
const AnalyticsCard = ({ title, value, icon, color }) => {
    return (
        <div className="bg-white overflow-hidden shadow-xl rounded-xl transition-all duration-300 hover:shadow-2xl border border-gray-200">
            <div className="p-5 flex items-start justify-between">
                <div className="w-0 flex-1">
                    <dt className="text-sm font-medium text-gray-500 truncate uppercase tracking-wide">{title}</dt>
                    {/* Changed font-extrabold to font-semibold for less visual weight */}
                    <dd className="mt-1 text-3xl font-semibold text-gray-900">
                        {title.includes('Value') ? `${CURRENCY_SYMBOL}${value.toLocaleString()}` : value.toLocaleString()}
                    </dd>
                </div>
                <div className={`flex-shrink-0 p-3 rounded-full ${color.bg} shadow-lg`}>
                    {React.cloneElement(icon, { className: 'h-6 w-6 text-white', 'aria-hidden': 'true' })}
                </div>
            </div>
        </div>
    );
};

// --- Sub-Component: Table Header ---
const TableHeader = () => (
    <thead className="bg-gray-50 border-t border-gray-200">
        <tr>
            <th scope="col" className="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">
                Product Name
            </th>
            <th scope="col" className="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">
                Price
            </th>
            <th scope="col" className="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">
                Inventory Status
            </th>
            <th scope="col" className="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">
                Category
            </th>
            <th scope="col" className="relative px-6 py-3 text-center">
                <span className="text-xs font-semibold text-gray-600 uppercase tracking-wider">Actions</span>
            </th>
        </tr>
    </thead>
);

// --- Sub-Component: Table Row with Stock Status ---
const ProductTableRow = ({ product }) => {
    const getStockStatus = (stock) => {
        if (product.is_digital) {
            return { text: 'Digital / Available', color: 'bg-indigo-100 text-indigo-800 ring-indigo-500/10', quantity: 'N/A' };
        }
        
        const stockInt = parseInt(stock || 0);
        if (stockInt > 10) {
            return { text: 'In Stock', color: 'bg-green-100 text-green-800 ring-green-500/10', quantity: `(${stockInt})` };
        } else if (stockInt > 0) {
            return { text: 'Low Stock', color: 'bg-yellow-100 text-yellow-800 ring-yellow-500/10', quantity: `(${stockInt})` };
        } else {
            return { text: 'Out of Stock', color: 'bg-red-100 text-red-800 ring-red-500/10', quantity: '(0)' };
        }
    };

    const stockStatus = getStockStatus(product.stock);

    return (
        <tr className="bg-white hover:bg-gray-50/70 transition-colors duration-150">
            <td className="px-6 py-4">
                <div className="flex items-center">
                    <div className="flex-shrink-0 h-10 w-10 mr-3 rounded-lg bg-gray-100 border border-gray-200 flex items-center justify-center">
                        <CubeIcon className="w-5 h-5 text-gray-400" />
                    </div>
                    <div className="flex flex-col text-sm font-medium text-gray-900 truncate max-w-xs">
                        <span className="font-semibold">{product.name}</span>
                        <span className="text-xs text-gray-400 mt-0.5">ID: {product.external_id}</span>
                    </div>
                </div>
            </td>
            <td className="px-6 py-4 whitespace-nowrap text-lg font-bold text-gray-900">
                {CURRENCY_SYMBOL}{parseFloat(product.price).toFixed(2)}
            </td>
            <td className="px-6 py-4 whitespace-nowrap">
                <span className={`inline-flex px-3 py-1 text-xs font-medium rounded-full ring-1 ring-inset ${stockStatus.color}`}>
                    {stockStatus.text} <span className="ml-1 font-bold">{stockStatus.quantity}</span>
                </span>
            </td>
            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-600">
                <span className="bg-emerald-50 text-emerald-700 px-3 py-1 rounded-full text-xs font-medium border border-emerald-200">
                    {product.category || 'Uncategorized'}
                </span>
            </td>
            <td className="px-6 py-4 whitespace-nowrap text-center text-sm font-medium">
                <div className="flex justify-center space-x-3">
                    <Link
                        href={route('admin.products.edit', product.external_id)}
                        title="Edit Product"
                        className="text-gray-500 hover:text-emerald-600 p-2 rounded-full hover:bg-emerald-50 transition duration-150"
                    >
                        <PencilIcon className="w-5 h-5" />
                    </Link>
                    <button
                        onClick={() => { /* Implement delete modal with Inertia.delete */ }}
                        title="Delete Product"
                        className="text-gray-500 hover:text-red-600 p-2 rounded-full hover:bg-red-50 transition duration-150"
                    >
                        <TrashIcon className="w-5 h-5" />
                    </button>
                </div>
            </td>
        </tr>
    );
};

// --- Sub-Component: Pagination (unchanged) ---
const Pagination = ({ links }) => {
    if (!links || links.length <= 3) return null; 
    
    const firstLink = links[0];
    const lastLink = links[links.length - 1];
    const relevantLinks = links.slice(1, -1);

    return (
        <div className="flex items-center justify-between">
            <div>
                <p className="text-sm text-gray-600">
                    Showing <span className="font-semibold text-gray-900">1</span> to <span className="font-semibold text-gray-900">10</span> of{' '}
                    <span className="font-semibold text-gray-900">97</span> results
                </p>
            </div>
            
            <div>
                <nav className="relative z-0 inline-flex rounded-lg shadow-sm" aria-label="Pagination">
                    
                    <Link
                        href={firstLink.url || '#'}
                        preserveScroll
                        className={`relative inline-flex items-center px-2 py-2 rounded-l-lg border border-gray-300 bg-white text-sm font-medium transition duration-150 ${
                            !firstLink.url ? 'cursor-not-allowed text-gray-400' : 'text-gray-700 hover:bg-gray-50'
                        }`}
                        aria-disabled={!firstLink.url}
                    >
                        <span className="sr-only">Previous</span>
                        <ChevronLeftIcon className="h-5 w-5" aria-hidden="true" />
                    </Link>

                    {relevantLinks.map((link, key) => (
                        <Link
                            key={key}
                            href={link.url || '#'}
                            preserveScroll
                            className={`hidden md:inline-flex items-center px-4 py-2 border text-sm font-medium transition duration-150 ${
                                link.active
                                    ? 'z-10 bg-emerald-50 border-emerald-500 text-emerald-700 font-bold' 
                                    : 'bg-white border-gray-300 text-gray-700 hover:bg-gray-50'
                            } ${
                                !link.url ? 'cursor-not-allowed opacity-50' : ''
                            } -ml-px`}
                            dangerouslySetInnerHTML={{ __html: link.label }}
                        />
                    ))}

                    <Link
                        href={lastLink.url || '#'}
                        preserveScroll
                        className={`relative inline-flex items-center px-2 py-2 rounded-r-lg border border-gray-300 bg-white text-sm font-medium transition duration-150 ${
                            !lastLink.url ? 'cursor-not-allowed text-gray-400' : 'text-gray-700 hover:bg-gray-50'
                        } -ml-px`}
                        aria-disabled={!lastLink.url}
                    >
                        <span className="sr-only">Next</span>
                        <ChevronRightIcon className="h-5 w-5" aria-hidden="true" />
                    </Link>
                </nav>
            </div>
        </div>
    );
};

// --- Main Component ---
export default function Index({ auth, products }) {
    const [searchTerm, setSearchTerm] = useState('');

    
    const productsData = products?.data?.length > 0 ? products.data : [];
    const paginationLinks = products?.links || [
        { url: null, label: '&laquo; Previous', active: false },
        { url: '#', label: '1', active: true },
        { url: '#', label: '2', active: false },
        { url: '#', label: '3', active: false },
        { url: '#', label: 'Next &raquo;', active: false },
    ];
    const analytics = getProductAnalytics(productsData);

    const cardColors = {
        total: { bg: 'bg-indigo-500' },
        value: { bg: 'bg-emerald-600' },
        low: { bg: 'bg-yellow-500' },
        out: { bg: 'bg-red-500' },
    };
    
    const handleSearch = (e) => {
        e.preventDefault();
        // In a real Inertia app: router.get(route('admin.products.index'), { search: searchTerm }, { preserveState: true });
        console.log('Searching for:', searchTerm);
    };

    return (
        <AdminLayout
            user={auth.user}
            header={
                <h1 className="text-3xl font-extrabold text-gray-900 flex items-center">
                    <ChartBarIcon className="w-8 h-8 mr-3 text-emerald-600" />
                    Inventory & Product Analytics
                </h1>
            }
        >
            <Head title="Product Dashboard" />

            <div className="py-8">
                <div className="mx-auto sm:px-6 lg:px-8">
                    
                    {/* --- 1. Analytics Section (Corporate View) --- */}
                    <div className="grid grid-cols-1 gap-6 sm:grid-cols-2 lg:grid-cols-4 mb-8">
                        <AnalyticsCard
                            title="Total Products"
                            value={analytics.totalProducts}
                            icon={<Bars3CenterLeftIcon />}
                            color={cardColors.total}
                        />
                        <AnalyticsCard
                            title="Inventory Value"
                            value={analytics.totalInventoryValue}
                            icon={<ShieldCheckIcon />}
                            color={cardColors.value}
                        />
                        <AnalyticsCard
                            title="Low Stock Items"
                            value={analytics.lowStockCount}
                            icon={<ExclamationTriangleIcon />}
                            color={cardColors.low}
                        />
                         <AnalyticsCard
                            title="Out of Stock Items"
                            value={analytics.outOfStockCount}
                            icon={<CubeIcon />}
                            color={cardColors.out}
                        />
                    </div>
                    
                    {/* --- 2. Product Management Table --- */}
                    <div className="bg-white overflow-hidden shadow-2xl rounded-xl border border-gray-100">
                        
                        {/* Header and Action Bar */}
                        <div className="p-6 border-b border-gray-200 flex flex-col sm:flex-row sm:items-center sm:justify-between">
                            <h3 className="text-xl font-semibold text-gray-800 mb-4 sm:mb-0">Product Catalog</h3>
                            
                            <Link
                                href={route('admin.products.create')}
                                className="inline-flex items-center px-4 py-2 bg-emerald-600 border border-transparent rounded-lg font-semibold text-sm text-white tracking-wider hover:bg-emerald-700 active:bg-emerald-900 focus:outline-none focus:ring-2 focus:ring-emerald-500 focus:ring-offset-2 transition ease-in-out duration-150 shadow-md"
                            >
                                <PlusIcon className="w-5 h-5 mr-2" />
                                Add New Product
                            </Link>
                        </div>
                        
                        {/* Search Bar */}
                        <div className="px-6 py-4 border-b border-gray-200">
                            <form onSubmit={handleSearch} className="relative max-w-lg">
                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <MagnifyingGlassIcon className="h-5 w-5 text-gray-400" />
                                </div>
                                <input
                                    type="search"
                                    className="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-lg leading-5 bg-white placeholder-gray-500 focus:outline-none focus:ring-1 focus:ring-emerald-500 focus:border-emerald-500 sm:text-sm shadow-sm transition-colors"
                                    placeholder="Search for a product by name or ID..."
                                    value={searchTerm}
                                    onChange={(e) => setSearchTerm(e.target.value)}
                                />
                            </form>
                        </div>

                        {/* Product Table */}
                        <div className="overflow-x-auto">
                            <table className="min-w-full divide-y divide-gray-200">
                                <TableHeader />
                                <tbody className="bg-white divide-y divide-gray-100">
                                    {productsData.map((product) => (
                                        <ProductTableRow key={product.id} product={product} />
                                    ))}
                                </tbody>
                            </table>
                            
                            {/* Empty State */}
                            {productsData.length === 0 && (
                                <div className="text-center py-16">
                                    <CubeIcon className="mx-auto h-12 w-12 text-gray-400" />
                                    <h3 className="mt-2 text-lg font-medium text-gray-900">No Products Found</h3>
                                    <p className="mt-1 text-sm text-gray-500">
                                        Get started by adding a new product.
                                    </p>
                                    <div className="mt-6">
                                        <Link
                                            href={route('admin.products.create')}
                                            className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-emerald-600 hover:bg-emerald-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500"
                                        >
                                            <PlusIcon className="-ml-1 mr-2 h-5 w-5" aria-hidden="true" />
                                            Add New Product
                                        </Link>
                                    </div>
                                </div>
                            )}
                        </div>

                        {/* Footer and Pagination */}
                        <div className="px-6 py-4 bg-gray-50 border-t border-gray-200">
                            <Pagination links={paginationLinks} />
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}