import React from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link } from '@inertiajs/react';
import { 
    PencilIcon, 
    TrashIcon, 
    PlusCircleIcon, 
    CurrencyDollarIcon,
    BeakerIcon,
    GiftIcon,
    ChartBarIcon,
    ClockIcon
} from '@heroicons/react/24/outline';

// --- Helper function to calculate analytics ---
const getAnalytics = (services) => {
    const totalServices = services.length;
    const freeServices = services.filter(s => s.price === 0).length;
    const paidServices = services.filter(s => s.price > 0).length;
    
    const averagePrice = paidServices > 0 
        ? (services.reduce((sum, s) => sum + parseFloat(s.price || 0), 0) / paidServices).toFixed(2)
        : '0.00';

    const mostRecentService = services.sort((a, b) => new Date(b.created_at) - new Date(a.created_at))[0];

    return { totalServices, freeServices, paidServices, averagePrice, mostRecentService };
};

// --- Sub-Component: Analytics Card ---
const AnalyticsCard = ({ title, value, icon, color }) => {
    return (
        <div className="bg-white overflow-hidden shadow-lg rounded-lg">
            <div className="p-5">
                <div className="flex items-center">
                    <div className={`flex-shrink-0 p-3 rounded-md ${color}`}>
                        {icon}
                    </div>
                    <div className="ml-5 w-0 flex-1">
                        <dl>
                            <dt className="text-sm font-medium text-gray-500 truncate">{title}</dt>
                            <dd className="text-lg font-semibold text-gray-900">{value}</dd>
                        </dl>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default function Index({ auth, schoolServices }) {
    const analytics = getAnalytics(schoolServices.data);
    return (
        <AdminLayout
            user={auth.user}
            header={
                <div>
                    <h2 className="font-semibold text-xl text-gray-800 leading-tight">School Services</h2>
                    <p className="mt-1 text-sm text-gray-600">Manage and overview all services offered by your school.</p>
                </div>
            }
        >
            <Head title="School Services" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* --- Analytics Section --- */}
                    <div className="grid grid-cols-1 gap-5 sm:grid-cols-2 lg:grid-cols-4 mb-8">
                        <AnalyticsCard
                            title="Total Services"
                            value={analytics.totalServices}
                            icon={<BeakerIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-slate-500"
                        />
                        <AnalyticsCard
                            title="Free Services"
                            value={analytics.freeServices}
                            icon={<GiftIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-emerald-500"
                        />
                        <AnalyticsCard
                            title="Paid Services"
                            value={analytics.paidServices}
                            icon={<CurrencyDollarIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-blue-500"
                        />
                        <AnalyticsCard
                            title="Avg. Price"
                            value="0"
                            icon={<ChartBarIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-indigo-500"
                        />
                    </div>

                    {/* Header with Create Button */}
                    <div className="flex justify-between items-center mb-6">
                        <h3 className="text-2xl font-bold text-gray-900">All Services</h3>
                        <Link
                            href={route('admin.school-services.create')}
                            className="inline-flex items-center px-4 py-2 bg-emerald-600 border border-transparent rounded-md font-semibold text-sm text-white hover:bg-emerald-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500 transition-all duration-200 shadow-sm"
                        >
                            <PlusCircleIcon className="-ml-1 mr-2 h-5 w-5" />
                            Add New Service
                        </Link>
                    </div>

                    {/* Services List */}
                    <div className="bg-white shadow-lg rounded-lg overflow-hidden">
                        {schoolServices.data.length > 0 ? (
                            <ul className="divide-y divide-gray-200">
                                {schoolServices.data.map((service) => (
                                    <li key={service.id} className="hover:bg-gray-50 transition-colors duration-150">
                                        <div className="px-6 py-5">
                                            <div className="flex items-center justify-between">
                                                <div className="flex-1 min-w-0">
                                                    {/* Service Name and Description */}
                                                    <p className="text-sm font-medium text-gray-900 truncate">{service.name}</p>
                                                    <p className="mt-1 text-sm text-gray-500 line-clamp-2">{service.description}</p>
                                                </div>
                                                <div className="flex items-center space-x-4 ml-4">
                                                    {/* Price Badge */}
                                                    <div className="flex-shrink-0">
                                                        {service.price > 0 ? (
                                                            <span className="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-emerald-100 text-emerald-800">
                                                                <CurrencyDollarIcon className="w-4 h-4 mr-1" />
                                                                ₵{service.price}
                                                            </span>
                                                        ) : (
                                                            <span className="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-gray-100 text-gray-800">
                                                                Free
                                                            </span>
                                                        )}
                                                    </div>
                                                    {/* Action Buttons */}
                                                    <div className="flex-shrink-0 flex space-x-2">
                                                        <Link
                                                            href={route('admin.school-services.edit', service.id)}
                                                            className="inline-flex items-center p-2 border border-gray-300 rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500"
                                                        >
                                                            <PencilIcon className="h-4 w-4" aria-hidden="true" />
                                                            <span className="sr-only">Edit {service.name}</span>
                                                        </Link>
                                                        <button
                                                            onClick={() => { /* Implement delete functionality */ }}
                                                            className="inline-flex items-center p-2 border border-transparent rounded-md text-red-700 bg-red-100 hover:bg-red-200 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500"
                                                        >
                                                            <TrashIcon className="h-4 w-4" aria-hidden="true" />
                                                            <span className="sr-only">Delete {service.name}</span>
                                                        </button>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </li>
                                ))}
                            </ul>
                        ) : (
                            // Empty State
                            <div className="text-center py-12">
                                <BeakerIcon className="mx-auto h-12 w-12 text-gray-400" />
                                <h3 className="mt-2 text-sm font-medium text-gray-900">No services</h3>
                                <p className="mt-1 text-sm text-gray-500">Get started by creating a new school service.</p>
                                <div className="mt-6">
                                    <Link
                                        href={route('admin.school-services.create')}
                                        className="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-emerald-600 hover:bg-emerald-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500"
                                    >
                                        <PlusCircleIcon className="-ml-1 mr-2 h-5 w-5" aria-hidden="true" />
                                        New Service
                                    </Link>
                                </div>
                            </div>
                        )}
                    </div>

                    {/* Pagination */}
                    {schoolServices.links && schoolServices.links.length > 3 && (
                        <div className="mt-6 flex justify-center">
                            {schoolServices.links.map((link, key) => (
                                <Link
                                    key={key}
                                    href={link.url || '#'}
                                    className={`mr-1 mb-1 px-4 py-3 text-sm leading-4 border rounded-md transition-colors duration-150 ${
                                        link.active
                                            ? 'bg-emerald-600 text-white border-emerald-600'
                                            : 'bg-white text-gray-700 border-gray-300 hover:bg-gray-50'
                                    } ${!link.url ? 'opacity-50 cursor-not-allowed' : ''}`}
                                    dangerouslySetInnerHTML={{ __html: link.label }}
                                />
                            ))}
                        </div>
                    )}
                </div>
            </div>
        </AdminLayout>
    );
}