import { Link, Head } from '@inertiajs/react';
import { 
    BeakerIcon, 
    CurrencyDollarIcon, 
    AcademicCapIcon, 
    UserGroupIcon,
    PencilIcon,
    TrashIcon
} from '@heroicons/react/24/outline';
import AdminLayout from '@/Layouts/AdminLayout';

// --- Helper function to calculate analytics ---
const getAnalytics = (services) => {
    const totalServices = services.length;
    const serviceCounts = services.reduce((acc, service) => {
        acc[service.type] = (acc[service.type] || 0) + 1;
        return acc;
    }, {});

    const mostRecentService = services.sort((a, b) => new Date(b.created_at) - new Date(a.created_at))[0];

    return { totalServices, serviceCounts, mostRecentService };
};

// --- Sub-Component: Analytics Card ---
const AnalyticsCard = ({ title, value, icon, color }) => {
    return (
        <div className="bg-white overflow-hidden shadow-lg rounded-lg">
            <div className="p-5">
                <div className="flex items-center">
                    <div className={`flex-shrink-0 p-3 rounded-md ${color}`}>
                        {icon}
                    </div>
                    <div className="ml-5 w-0 flex-1">
                        <dl>
                            <dt className="text-sm font-medium text-gray-500 truncate">{title}</dt>
                            <dd className="text-lg font-semibold text-gray-900">{value}</dd>
                        </dl>
                    </div>
                </div>
            </div>
        </div>
    );
};

// --- Sub-Component: Service Type Badge ---
const ServiceTypeBadge = ({ type }) => {
    const badgeStyles = {
        free: 'bg-green-100 text-green-800',
        paid: 'bg-blue-100 text-blue-800',
        workshop: 'bg-purple-100 text-purple-800',
        consultation: 'bg-yellow-100 text-yellow-800',
    };
    const style = badgeStyles[type.toLowerCase()] || 'bg-gray-100 text-gray-800';
    return <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${style}`}>{type}</span>;
};

export default function Index({ schoolServices }) {
    const analytics = getAnalytics(schoolServices.data);

    return (
        <AdminLayout>
            <div className="min-h-screen bg-gray-50">
                <Head title="Manage School Services" />

                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8 py-8">
                    {/* Page Header */}
                    <div className="md:flex md:items-center md:justify-between mb-8">
                        <div className="flex-1 min-w-0">
                            <h2 className="text-2xl font-bold leading-7 text-gray-900 sm:text-3xl sm:truncate">
                                School Services Dashboard
                            </h2>
                        </div>
                        <div className="mt-4 flex md:mt-0 md:ml-4">
                            <Link
                                href={route('admin.school-services.create')}
                                className="inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                            >
                                Create New Service
                            </Link>
                        </div>
                    </div>

                    {/* Analytics Section */}
                    <div className="grid grid-cols-1 gap-5 sm:grid-cols-2 lg:grid-cols-4 mb-8">
                        <AnalyticsCard
                            title="Total Services"
                            value={analytics.totalServices}
                            icon={<BeakerIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-green-500"
                        />
                        <AnalyticsCard
                            title="Free Services"
                            value={analytics.serviceCounts.free || 0}
                            icon={<CurrencyDollarIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-emerald-500"
                        />
                        <AnalyticsCard
                            title="Paid Services"
                            value={analytics.serviceCounts.paid || 0}
                            icon={<UserGroupIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-blue-500"
                        />
                        <AnalyticsCard
                            title="Most Recent"
                            value={analytics.mostRecentService?.name || 'N/A'}
                            icon={<AcademicCapIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-indigo-500"
                        />
                    </div>

                    {/* Services by Type Breakdown */}
                    <div className="bg-white shadow rounded-lg mb-8 p-6">
                        <h3 className="text-lg leading-6 font-medium text-gray-900 mb-4">Services by Type</h3>
                        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
                            {Object.entries(analytics.serviceCounts).map(([type, count]) => (
                                <div key={type} className="text-center p-4 bg-gray-50 rounded-lg">
                                    <p className="text-2xl font-bold text-gray-900">{count}</p>
                                    <ServiceTypeBadge type={type} />
                                </div>
                            ))}
                        </div>
                    </div>

                    {/* Services Table */}
                    <div className="bg-white shadow overflow-hidden sm:rounded-md">
                        <ul className="divide-y divide-gray-200">
                            {schoolServices.data.length > 0 ? (
                                schoolServices.data.map((service) => (
                                    <li key={service.id}>
                                        <div className="block hover:bg-gray-50">
                                            <div className="px-4 py-4 sm:px-6">
                                                <div className="flex items-center justify-between">
                                                    <div className="flex-1 min-w-0">
                                                        <p className="text-sm font-medium text-gray-900 truncate">{service.name}</p>
                                                        <p className="mt-1 text-sm text-gray-500 line-clamp-2">{service.description}</p>
                                                    </div>
                                                    <div className="flex items-center space-x-4 ml-4">
                                                        <ServiceTypeBadge type={service.type} />
                                                        <div className="flex-shrink-0">
                                                            <Link
                                                                href={route('admin.school-services.edit', service.id)}
                                                                className="inline-flex items-center p-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50"
                                                            >
                                                                <PencilIcon className="h-4 w-4" aria-hidden="true" />
                                                            </Link>
                                                        </div>
                                                        <div className="flex-shrink-0">
                                                            <Link
                                                                href={route('admin.school-services.destroy', service.id)}
                                                                method="delete"
                                                                as="button"
                                                                type="button"
                                                                className="inline-flex items-center p-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-red-700 bg-red-100 hover:bg-red-200"
                                                            >
                                                                <TrashIcon className="h-4 w-4" aria-hidden="true" />
                                                            </Link>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </li>
                                ))
                            ) : (
                                <div className="text-center py-12">
                                    <AcademicCapIcon className="mx-auto h-12 w-12 text-gray-400" />
                                    <h3 className="mt-2 text-sm font-medium text-gray-900">No services</h3>
                                    <p className="mt-1 text-sm text-gray-500">Get started by creating a new school service.</p>
                                    <div className="mt-6">
                                        <Link
                                            href={route('admin.school-services.create')}
                                            className="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                        >
                                            <BeakerIcon className="-ml-1 mr-2 h-5 w-5" aria-hidden="true" />
                                            New Service
                                        </Link>
                                    </div>
                                </div>
                            )}
                        </ul>
                    </div>

                    {/* Pagination */}
                    {schoolServices.links && schoolServices.links.length > 3 && (
                        <div className="mt-6 flex justify-center">
                            {schoolServices.links.map((link, key) => (
                                link.url === null ? (
                                    <span key={key} className="mr-1 mb-1 px-4 py-3 text-sm leading-4 text-gray-400 border rounded-md" dangerouslySetInnerHTML={{ __html: link.label }} />
                                ) : (
                                    <Link
                                        key={key}
                                        className={`mr-1 mb-1 px-4 py-3 text-sm leading-4 border rounded-md hover:bg-white focus:border-green-500 focus:text-green-500 ${link.active ? 'bg-green-500 text-white border-green-500' : 'bg-white text-gray-700'}`}
                                        href={link.url}
                                        dangerouslySetInnerHTML={{ __html: link.label }}
                                    />
                                )
                            ))}
                        </div>
                    )}
                </div>
            </div>
        </AdminLayout>
    );
}
