import AdminLayout from '@/Layouts/AdminLayout';
import { Head, useForm } from '@inertiajs/react';
import { useState, useEffect } from 'react';

export default function Edit({ setting, toast }) {
    const [formProgress, setFormProgress] = useState(0);
    const [activeSection, setActiveSection] = useState('company');
    const [previewMode, setPreviewMode] = useState(false);
    
    const { data, setData, post, errors, processing, reset } = useForm({
        _method: 'POST',
        logo: null,
        company_name: setting.company_name || '',
        email: setting.email || '',
        phone: setting.phone || '',
        address: setting.address || '',
        facebook_link: setting.facebook_link || '',
        twitter_link: setting.twitter_link || '',
        linkedin_link: setting.linkedin_link || '',
        instagram_link: setting.instagram_link || '',
        youtube_link: setting.youtube_link || '',
        years_of_experience: setting.years_of_experience || '',
        happy_clients: setting.happy_clients || '',
        projects_completion: setting.projects_completion || '',
        about_us_description: setting.about_us_description || '',
    });

    // Calculate form completion progress
    useEffect(() => {
        const fields = [
            'company_name', 'email', 'phone', 'address', 
            'about_us_description', 'years_of_experience', 
            'happy_clients', 'projects_completion'
        ];
        
        const filledFields = fields.filter(field => data[field] && data[field].trim() !== '');
        const progress = Math.round((filledFields.length / fields.length) * 100);
        setFormProgress(progress);
    }, [data]);

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route('admin.settings.update', { setting: setting.id }), {
            onSuccess: () => {
                toast.fire({
                    icon: 'success',
                    title: 'Settings updated successfully.'
                });
                // Reset form data after successful submission
                reset('logo');
            },
            onError: () => {
                toast.fire({
                    icon: 'error',
                    title: 'Please check the form for errors.'
                });
            }
        });
    };

    const handleCancel = () => {
        // Reset form to original values
        reset();
        toast.fire({
            icon: 'info',
            title: 'Changes discarded.'
        });
    };

    const sections = [
        { id: 'company', name: 'Company Info', icon: 'M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4' },
        { id: 'contact', name: 'Contact Info', icon: 'M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z' },
        { id: 'social', name: 'Social Media', icon: 'M13.828 10.172a4 4 0 00-5.656 0l-4 4a4 4 0 105.656 5.656l1.102-1.101m-.758-4.899a4 4 0 005.656 0l4-4a4 4 0 00-5.656-5.656l-1.1 1.1' },
        { id: 'stats', name: 'Statistics', icon: 'M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z' }
    ];

    return (
        <AdminLayout
            header={
                <div className="flex items-center justify-between">
                    <h2 className="text-xl font-semibold leading-tight text-gray-800">Company Settings</h2>
                    <div className="flex items-center space-x-2 text-sm text-gray-600">
                        <span>Dashboard</span>
                        <span>/</span>
                        <span>Settings</span>
                    </div>
                </div>
            }
        >
            <Head title="Company Settings" />

            <div className="py-8">
                <div className="mx-auto max-w-7xl space-y-6 sm:px-6 lg:px-8">
                    {/* Page Header with Progress Bar */}
                    <div className="bg-gradient-to-r from-green-600 to-emerald-700 rounded-xl p-6 text-white shadow-lg">
                        <h1 className="text-2xl font-bold">Company Configuration</h1>
                        <p className="mt-2 opacity-90">Manage your company information, contact details, and statistics</p>
                        
                        {/* Progress Bar */}
                        <div className="mt-4">
                            <div className="flex justify-between text-sm mb-1">
                                <span>Profile Completion</span>
                                <span>{formProgress}%</span>
                            </div>
                            <div className="w-full bg-white/20 rounded-full h-2">
                                <div 
                                    className="bg-white h-2 rounded-full transition-all duration-500 ease-out"
                                    style={{ width: `${formProgress}%` }}
                                ></div>
                            </div>
                        </div>
                    </div>

                    {/* Section Navigation */}
                    <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-1 flex flex-wrap">
                        {sections.map((section) => (
                            <button
                                key={section.id}
                                onClick={() => setActiveSection(section.id)}
                                className={`flex items-center px-4 py-2 rounded-lg text-sm font-medium transition-all duration-200 ${
                                    activeSection === section.id
                                        ? 'bg-green-100 text-green-700'
                                        : 'text-gray-600 hover:text-gray-900 hover:bg-gray-100'
                                }`}
                            >
                                <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d={section.icon} />
                                </svg>
                                {section.name}
                            </button>
                        ))}
                    </div>

                    <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                        {/* Main Form */}
                        <div className="lg:col-span-2">
                            <form onSubmit={handleSubmit} className="space-y-6">
                                {/* Company Information Section */}
                                <div className={`bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden transition-all duration-300 ${activeSection === 'company' ? 'ring-2 ring-green-500 ring-opacity-50' : ''}`}>
                                    <div className="bg-gray-50 px-6 py-4 border-b border-gray-200">
                                        <h3 className="text-lg font-medium text-gray-800 flex items-center">
                                            <svg className="w-5 h-5 mr-2 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                                            </svg>
                                            Company Information
                                        </h3>
                                    </div>
                                    <div className="p-6 space-y-4">
                                        <div>
                                            <label htmlFor="company_name" className="block text-sm font-medium text-gray-700 mb-1">Company Name</label>
                                            <input
                                                type="text"
                                                id="company_name"
                                                value={data.company_name}
                                                onChange={(e) => setData('company_name', e.target.value)}
                                                className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                    errors.company_name ? 'border-red-300' : 'border-gray-300'
                                                }`}
                                                placeholder="Enter company name"
                                            />
                                            {errors.company_name && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.company_name}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label htmlFor="logo" className="block text-sm font-medium text-gray-700 mb-1">Company Logo</label>
                                            <div className="flex items-center space-x-4">
                                                <div className="flex-shrink-0">
                                                    {setting.logo_path ? (
                                                        <img className="h-16 w-16 rounded-md object-cover border border-gray-300" src={`/storage/${setting.logo_path}`} alt="Company Logo" />
                                                    ) : (
                                                        <div className="h-16 w-16 rounded-md bg-gray-100 flex items-center justify-center border border-gray-300">
                                                            <svg className="h-8 w-8 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                                                            </svg>
                                                        </div>
                                                    )}
                                                </div>
                                                <div className="flex-1">
                                                    <label className="block">
                                                        <span className="sr-only">Choose logo file</span>
                                                        <input
                                                            type="file"
                                                            id="logo"
                                                            onChange={(e) => setData('logo', e.target.files[0])}
                                                            className="mt-1 block w-full text-sm text-gray-900 border border-gray-300 rounded-lg cursor-pointer bg-gray-50 focus:outline-none focus:border-green-500"
                                                        />
                                                    </label>
                                                    <p className="mt-1 text-xs text-gray-500">JPG, PNG or GIF. Max 1MB.</p>
                                                    {errors.logo && (
                                                        <p className="mt-1 text-sm text-red-600 flex items-center">
                                                            <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                                <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                            </svg>
                                                            {errors.logo}
                                                        </p>
                                                    )}
                                                </div>
                                            </div>
                                        </div>

                                        <div>
                                            <label htmlFor="about_us_description" className="block text-sm font-medium text-gray-700 mb-1">About Us Description</label>
                                            <textarea
                                                id="about_us_description"
                                                value={data.about_us_description}
                                                onChange={(e) => setData('about_us_description', e.target.value)}
                                                rows={4}
                                                className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                    errors.about_us_description ? 'border-red-300' : 'border-gray-300'
                                                }`}
                                                placeholder="Describe your company..."
                                            ></textarea>
                                            <div className="flex justify-between mt-1">
                                                <span className="text-xs text-gray-500">Brief description of your company</span>
                                                <span className="text-xs text-gray-500">{data.about_us_description.length}/500</span>
                                            </div>
                                            {errors.about_us_description && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.about_us_description}
                                                </p>
                                            )}
                                        </div>
                                    </div>
                                </div>

                                {/* Contact Information Section */}
                                <div className={`bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden transition-all duration-300 ${activeSection === 'contact' ? 'ring-2 ring-green-500 ring-opacity-50' : ''}`}>
                                    <div className="bg-gray-50 px-6 py-4 border-b border-gray-200">
                                        <h3 className="text-lg font-medium text-gray-800 flex items-center">
                                            <svg className="w-5 h-5 mr-2 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
                                            </svg>
                                            Contact Information
                                        </h3>
                                    </div>
                                    <div className="p-6 grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <label htmlFor="email" className="block text-sm font-medium text-gray-700 mb-1">Email</label>
                                            <div className="relative">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <svg className="h-5 w-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
                                                    </svg>
                                                </div>
                                                <input
                                                    type="email"
                                                    id="email"
                                                    value={data.email}
                                                    onChange={(e) => setData('email', e.target.value)}
                                                    className={`mt-1 block w-full pl-10 rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.email ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="contact@company.com"
                                                />
                                            </div>
                                            {errors.email && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.email}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label htmlFor="phone" className="block text-sm font-medium text-gray-700 mb-1">Phone</label>
                                            <div className="relative">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <svg className="h-5 w-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
                                                    </svg>
                                                </div>
                                                <input
                                                    type="text"
                                                    id="phone"
                                                    value={data.phone}
                                                    onChange={(e) => setData('phone', e.target.value)}
                                                    className={`mt-1 block w-full pl-10 rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.phone ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="+1 (555) 123-4567"
                                                />
                                            </div>
                                            {errors.phone && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.phone}
                                                </p>
                                            )}
                                        </div>

                                        <div className="md:col-span-2">
                                            <label htmlFor="address" className="block text-sm font-medium text-gray-700 mb-1">Address</label>
                                            <div className="relative">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <svg className="h-5 w-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
                                                    </svg>
                                                </div>
                                                <input
                                                    type="text"
                                                    id="address"
                                                    value={data.address}
                                                    onChange={(e) => setData('address', e.target.value)}
                                                    className={`mt-1 block w-full pl-10 rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.address ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="123 Business Street, City, Country"
                                                />
                                            </div>
                                            {errors.address && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.address}
                                                </p>
                                            )}
                                        </div>
                                    </div>
                                </div>

                                {/* Social Media Section */}
                                <div className={`bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden transition-all duration-300 ${activeSection === 'social' ? 'ring-2 ring-green-500 ring-opacity-50' : ''}`}>
                                    <div className="bg-gray-50 px-6 py-4 border-b border-gray-200">
                                        <h3 className="text-lg font-medium text-gray-800 flex items-center">
                                            <svg className="w-5 h-5 mr-2 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13.828 10.172a4 4 0 00-5.656 0l-4 4a4 4 0 105.656 5.656l1.102-1.101m-.758-4.899a4 4 0 005.656 0l4-4a4 4 0 00-5.656-5.656l-1.1 1.1" />
                                            </svg>
                                            Social Media Links
                                        </h3>
                                    </div>
                                    <div className="p-6 grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <label htmlFor="facebook_link" className="block text-sm font-medium text-gray-700 mb-1">Facebook</label>
                                            <div className="relative">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <svg className="h-5 w-5 text-blue-600" fill="currentColor" viewBox="0 0 24 24">
                                                        <path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z" />
                                                    </svg>
                                                </div>
                                                <input
                                                    type="url"
                                                    id="facebook_link"
                                                    value={data.facebook_link}
                                                    onChange={(e) => setData('facebook_link', e.target.value)}
                                                    className={`mt-1 block w-full pl-10 rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.facebook_link ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="https://facebook.com/yourcompany"
                                                />
                                            </div>
                                            {errors.facebook_link && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.facebook_link}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label htmlFor="twitter_link" className="block text-sm font-medium text-gray-700 mb-1">Twitter</label>
                                            <div className="relative">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <svg className="h-5 w-5 text-blue-400" fill="currentColor" viewBox="0 0 24 24">
                                                        <path d="M23.953 4.57a10 10 0 01-2.825.775 4.958 4.958 0 002.163-2.723c-.951.555-2.005.959-3.127 1.184a4.92 4.92 0 00-8.384 4.482C7.69 8.095 4.067 6.13 1.64 3.162a4.822 4.822 0 00-.666 2.475c0 1.71.87 3.213 2.188 4.096a4.904 4.904 0 01-2.228-.616v.06a4.923 4.923 0 003.946 4.827 4.996 4.996 0 01-2.212.085 4.936 4.936 0 004.604 3.417 9.867 9.867 0 01-6.102 2.105c-.39 0-.779-.023-1.17-.067a13.995 13.995 0 007.557 2.209c9.053 0 13.998-7.496 13.998-13.985 0-.21 0-.42-.015-.63A9.935 9.935 0 0024 4.59z" />
                                                    </svg>
                                                </div>
                                                <input
                                                    type="url"
                                                    id="twitter_link"
                                                    value={data.twitter_link}
                                                    onChange={(e) => setData('twitter_link', e.target.value)}
                                                    className={`mt-1 block w-full pl-10 rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.twitter_link ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="https://twitter.com/yourcompany"
                                                />
                                            </div>
                                            {errors.twitter_link && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.twitter_link}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label htmlFor="linkedin_link" className="block text-sm font-medium text-gray-700 mb-1">LinkedIn</label>
                                            <div className="relative">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <svg className="h-5 w-5 text-blue-700" fill="currentColor" viewBox="0 0 24 24">
                                                        <path d="M19 0h-14c-2.761 0-5 2.239-5 5v14c0 2.761 2.239 5 5 5h14c2.762 0 5-2.239 5-5v-14c0-2.761-2.238-5-5-5zm-11 19h-3v-11h3v11zm-1.5-12.268c-.966 0-1.75-.79-1.75-1.764s.784-1.764 1.75-1.764 1.75.79 1.75 1.764-.783 1.764-1.75 1.764zm13.5 12.268h-3v-5.604c0-3.368-4-3.113-4 0v5.604h-3v-11h3v1.765c1.396-2.586 7-2.777 7 2.476v6.759z" />
                                                    </svg>
                                                </div>
                                                <input
                                                    type="url"
                                                    id="linkedin_link"
                                                    value={data.linkedin_link}
                                                    onChange={(e) => setData('linkedin_link', e.target.value)}
                                                    className={`mt-1 block w-full pl-10 rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.linkedin_link ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="https://linkedin.com/company/yourcompany"
                                                />
                                            </div>
                                            {errors.linkedin_link && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.linkedin_link}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label htmlFor="instagram_link" className="block text-sm font-medium text-gray-700 mb-1">Instagram</label>
                                            <div className="relative">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <svg className="h-5 w-5 text-pink-600" fill="currentColor" viewBox="0 0 24 24">
                                                        <path d="M12 2.163c3.204 0 3.584.012 4.85.07 3.252.148 4.771 1.691 4.919 4.919.058 1.265.069 1.645.069 4.849 0 3.205-.012 3.584-.069 4.849-.149 3.225-1.664 4.771-4.919 4.919-1.266.058-1.644.07-4.85.07-3.204 0-3.584-.012-4.849-.07-3.26-.149-4.771-1.699-4.919-4.92-.058-1.265-.07-1.644-.07-4.849 0-3.204.013-3.583.07-4.849.149-3.227 1.664-4.771 4.919-4.919 1.266-.057 1.645-.069 4.849-.069zm0-2.163c-3.259 0-3.667.014-4.947.072-4.358.2-6.78 2.618-6.98 6.98-.059 1.281-.073 1.689-.073 4.948 0 3.259.014 3.668.072 4.948.2 4.358 2.618 6.78 6.98 6.98 1.281.058 1.689.072 4.948.072 3.259 0 3.668-.014 4.948-.072 4.354-.2 6.782-2.618 6.979-6.98.059-1.28.073-1.689.073-4.948 0-3.259-.014-3.667-.072-4.947-.196-4.354-2.617-6.78-6.979-6.98-1.281-.059-1.69-.073-4.949-.073zm0 5.838c-3.403 0-6.162 2.759-6.162 6.162s2.759 6.163 6.162 6.163 6.162-2.759 6.162-6.163c0-3.403-2.759-6.162-6.162-6.162zm0 10.162c-2.209 0-4-1.79-4-4 0-2.209 1.791-4 4-4s4 1.791 4 4c0 2.21-1.791 4-4 4zm6.406-11.845c-.796 0-1.441.645-1.441 1.44s.645 1.44 1.441 1.44c.795 0 1.439-.645 1.439-1.44s-.644-1.44-1.439-1.44z" />
                                                    </svg>
                                                </div>
                                                <input
                                                    type="url"
                                                    id="instagram_link"
                                                    value={data.instagram_link}
                                                    onChange={(e) => setData('instagram_link', e.target.value)}
                                                    className={`mt-1 block w-full pl-10 rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.instagram_link ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="https://instagram.com/yourcompany"
                                                />
                                            </div>
                                            {errors.instagram_link && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.instagram_link}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label htmlFor="youtube_link" className="block text-sm font-medium text-gray-700 mb-1">YouTube</label>
                                            <div className="relative">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <svg className="h-5 w-5 text-red-600" fill="currentColor" viewBox="0 0 24 24">
                                                        <path d="M23.498 6.186a3.016 3.016 0 0 0-2.122-2.136C19.505 3.545 12 3.545 12 3.545s-7.505 0-9.377.505A3.017 3.017 0 0 0 .502 6.186C0 8.07 0 12 0 12s0 3.93.502 5.814a3.016 3.016 0 0 0 2.122 2.136c1.871.505 9.376.505 9.376.505s7.505 0 9.377-.505a3.015 3.015 0 0 0 2.122-2.136C24 15.93 24 12 24 12s0-3.93-.502-5.814zM9.545 15.568V8.432L15.818 12l-6.273 3.568z" />
                                                    </svg>
                                                </div>
                                                <input
                                                    type="url"
                                                    id="youtube_link"
                                                    value={data.youtube_link}
                                                    onChange={(e) => setData('youtube_link', e.target.value)}
                                                    className={`mt-1 block w-full pl-10 rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.youtube_link ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="https://youtube.com/c/yourcompany"
                                                />
                                            </div>
                                            {errors.youtube_link && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.youtube_link}
                                                </p>
                                            )}
                                        </div>
                                    </div>
                                </div>

                                {/* Statistics Section */}
                                <div className={`bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden transition-all duration-300 ${activeSection === 'stats' ? 'ring-2 ring-green-500 ring-opacity-50' : ''}`}>
                                    <div className="bg-gray-50 px-6 py-4 border-b border-gray-200">
                                        <h3 className="text-lg font-medium text-gray-800 flex items-center">
                                            <svg className="w-5 h-5 mr-2 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                                            </svg>
                                            Company Statistics
                                        </h3>
                                    </div>
                                    <div className="p-6 grid grid-cols-1 md:grid-cols-3 gap-4">
                                        <div>
                                            <label htmlFor="years_of_experience" className="block text-sm font-medium text-gray-700 mb-1">Years of Experience</label>
                                            <div className="relative">
                                                <input
                                                    type="number"
                                                    id="years_of_experience"
                                                    value={data.years_of_experience}
                                                    onChange={(e) => setData('years_of_experience', e.target.value)}
                                                    className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.years_of_experience ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="10"
                                                    min="0"
                                                />
                                                <div className="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                                    <span className="text-gray-500 sm:text-sm">years</span>
                                                </div>
                                            </div>
                                            {errors.years_of_experience && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.years_of_experience}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label htmlFor="happy_clients" className="block text-sm font-medium text-gray-700 mb-1">Happy Clients</label>
                                            <div className="relative">
                                                <input
                                                    type="number"
                                                    id="happy_clients"
                                                    value={data.happy_clients}
                                                    onChange={(e) => setData('happy_clients', e.target.value)}
                                                    className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.happy_clients ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="500"
                                                    min="0"
                                                />
                                                <div className="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                                    <span className="text-gray-500 sm:text-sm">clients</span>
                                                </div>
                                            </div>
                                            {errors.happy_clients && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.happy_clients}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label htmlFor="projects_completion" className="block text-sm font-medium text-gray-700 mb-1">Projects Completed</label>
                                            <div className="relative">
                                                <input
                                                    type="number"
                                                    id="projects_completion"
                                                    value={data.projects_completion}
                                                    onChange={(e) => setData('projects_completion', e.target.value)}
                                                    className={`mt-1 block w-full rounded-md shadow-sm focus:ring-green-500 focus:border-green-500 sm:text-sm ${
                                                        errors.projects_completion ? 'border-red-300' : 'border-gray-300'
                                                    }`}
                                                    placeholder="1000"
                                                    min="0"
                                                />
                                                <div className="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                                    <span className="text-gray-500 sm:text-sm">projects</span>
                                                </div>
                                            </div>
                                            {errors.projects_completion && (
                                                <p className="mt-1 text-sm text-red-600 flex items-center">
                                                    <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                    </svg>
                                                    {errors.projects_completion}
                                                </p>
                                            )}
                                        </div>
                                    </div>
                                </div>

                                {/* Action Buttons */}
                                <div className="flex justify-end space-x-3">
                                    <button
                                        type="button"
                                        onClick={handleCancel}
                                        className="px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 transition-colors duration-200"
                                    >
                                        Cancel
                                    </button>
                                    <button
                                        type="submit"
                                        className="px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 disabled:opacity-50 transition-all duration-200 flex items-center"
                                        disabled={processing}
                                    >
                                        {processing ? (
                                            <>
                                                <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                                    <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                                </svg>
                                                Saving...
                                            </>
                                        ) : (
                                            <>
                                                <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                                                </svg>
                                                Save Changes
                                            </>
                                        )}
                                    </button>
                                </div>
                            </form>
                        </div>

                        {/* Sidebar */}
                        <div className="lg:col-span-1 space-y-6">
                            {/* Tips Card */}
                            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                                <h3 className="text-lg font-medium text-gray-800 mb-4 flex items-center">
                                    <svg className="w-5 h-5 mr-2 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                                    </svg>
                                    Tips & Best Practices
                                </h3>
                                <div className="space-y-4">
                                    <div className="flex items-start">
                                        <div className="flex-shrink-0">
                                            <div className="flex items-center justify-center h-6 w-6 rounded-md bg-green-100 text-green-600">
                                                <svg className="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                                </svg>
                                            </div>
                                        </div>
                                        <div className="ml-3">
                                            <p className="text-sm text-gray-700">Keep your company information up-to-date to build trust with customers.</p>
                                        </div>
                                    </div>
                                    <div className="flex items-start">
                                        <div className="flex-shrink-0">
                                            <div className="flex items-center justify-center h-6 w-6 rounded-md bg-green-100 text-green-600">
                                                <svg className="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                                </svg>
                                            </div>
                                        </div>
                                        <div className="ml-3">
                                            <p className="text-sm text-gray-700">Use a high-quality logo with a transparent background for best results.</p>
                                        </div>
                                    </div>
                                    <div className="flex items-start">
                                        <div className="flex-shrink-0">
                                            <div className="flex items-center justify-center h-6 w-6 rounded-md bg-green-100 text-green-600">
                                                <svg className="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                                </svg>
                                            </div>
                                        </div>
                                        <div className="ml-3">
                                            <p className="text-sm text-gray-700">Regularly update your statistics to showcase your company's growth.</p>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {/* Preview Card */}
                            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                                <div className="flex justify-between items-center mb-4">
                                    <h3 className="text-lg font-medium text-gray-800">Preview</h3>
                                    <button
                                        type="button"
                                        onClick={() => setPreviewMode(!previewMode)}
                                        className="text-sm text-green-600 hover:text-green-700 font-medium"
                                    >
                                        {previewMode ? 'Edit Mode' : 'Preview Mode'}
                                    </button>
                                </div>
                                <div className={`bg-gray-50 rounded-lg p-4 border border-gray-200 ${previewMode ? 'transform scale-95' : ''}`}>
                                    <div className="flex items-center mb-4">
                                        {setting.logo_path ? (
                                            <img className="h-12 w-12 rounded-md object-cover" src={`/storage/${setting.logo_path}`} alt="Company Logo" />
                                        ) : (
                                            <div className="h-12 w-12 rounded-md bg-gray-200 flex items-center justify-center">
                                                <svg className="h-6 w-6 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                                                </svg>
                                            </div>
                                        )}
                                        <div className="ml-3">
                                            <h4 className="text-sm font-medium text-gray-900">{data.company_name || 'Company Name'}</h4>
                                            <p className="text-xs text-gray-500">{data.email || 'email@company.com'}</p>
                                        </div>
                                    </div>
                                    <div className="grid grid-cols-3 gap-2 text-center">
                                        <div className="bg-white p-2 rounded border border-gray-200">
                                            <p className="text-xs font-medium text-gray-500">Experience</p>
                                            <p className="text-sm font-bold text-green-600">{data.years_of_experience || '0'} yrs</p>
                                        </div>
                                        <div className="bg-white p-2 rounded border border-gray-200">
                                            <p className="text-xs font-medium text-gray-500">Clients</p>
                                            <p className="text-sm font-bold text-green-600">{data.happy_clients || '0'}</p>
                                        </div>
                                        <div className="bg-white p-2 rounded border border-gray-200">
                                            <p className="text-xs font-medium text-gray-500">Projects</p>
                                            <p className="text-sm font-bold text-green-600">{data.projects_completion || '0'}</p>
                                        </div>
                                    </div>
                                    {data.about_us_description && (
                                        <div className="mt-4 pt-4 border-t border-gray-200">
                                            <p className="text-xs text-gray-600 line-clamp-3">{data.about_us_description}</p>
                                        </div>
                                    )}
                                </div>
                            </div>

                            {/* Quick Stats Card */}
                            <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl shadow-sm border border-green-200 p-6">
                                <h3 className="text-lg font-medium text-gray-800 mb-4">Profile Strength</h3>
                                <div className="flex items-center justify-center mb-4">
                                    <div className="relative">
                                        <svg className="w-32 h-32">
                                            <circle
                                                cx="64"
                                                cy="64"
                                                r="56"
                                                stroke="currentColor"
                                                strokeWidth="8"
                                                fill="none"
                                                className="text-green-200"
                                            ></circle>
                                            <circle
                                                cx="64"
                                                cy="64"
                                                r="56"
                                                stroke="currentColor"
                                                strokeWidth="8"
                                                fill="none"
                                                strokeDasharray={`${formProgress * 3.51} 351`}
                                                className="text-green-600 transform -rotate-90 origin-center"
                                            ></circle>
                                        </svg>
                                        <div className="absolute inset-0 flex items-center justify-center">
                                            <span className="text-2xl font-bold text-green-600">{formProgress}%</span>
                                        </div>
                                    </div>
                                </div>
                                <div className="text-center">
                                    <p className="text-sm text-gray-600">
                                        {formProgress < 50 && "Complete more fields to improve your profile"}
                                        {formProgress >= 50 && formProgress < 80 && "Good progress! Keep going"}
                                        {formProgress >= 80 && "Excellent! Your profile is almost complete"}
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}