import React, { useState } from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, useForm } from '@inertiajs/react';
import TextInput from '@/Components/TextInput';
import InputLabel from '@/Components/InputLabel';
import InputError from '@/Components/InputError';
import PrimaryButton from '@/Components/PrimaryButton';
import SecondaryButton from '@/Components/SecondaryButton';
import { Link } from '@inertiajs/react';

export default function TeamCreate() {
    const { data, setData, post, processing, errors, reset } = useForm({
        name: '',
        position: '',
        image: null,
        description: '',
        facebook_link: '',
        twitter_link: '',
        linkedin_link: '',
        order: 0,
        department: '',
        is_leadership: false,
    });
    
    const [imagePreview, setImagePreview] = useState(null);

    const handleImageChange = (e) => {
        const file = e.target.files[0];
        if (file) {
            setData('image', file);
            const reader = new FileReader();
            reader.onload = () => setImagePreview(reader.result);
            reader.readAsDataURL(file);
        }
    };

    const submit = (e) => {
        e.preventDefault();
        post(route('teams.store'), {
            onSuccess: () => reset(),
        });
    };

    // Common departments
    const departmentOptions = [
        'Executive', 'Management', 'Development', 'Design', 
        'Marketing', 'Sales', 'Support', 'HR', 'Finance', 'Other'
    ];

    return (
        <AdminLayout>
            <Head title="Create Team Member" />

            <div className="py-8 px-4 sm:px-6 lg:px-8">
                <div className="max-w-3xl mx-auto">
                    {/* Header Section */}
                    <div className="mb-8">
                        <div className="flex items-center mb-2">
                            <Link href={route('admin.teams.index')} className="text-sm font-medium text-indigo-600 hover:text-indigo-500 flex items-center">
                                <svg className="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"></path>
                                </svg>
                                Back to Team
                            </Link>
                        </div>
                        <h1 className="text-2xl font-bold text-gray-900">Add New Team Member</h1>
                        <p className="mt-1 text-sm text-gray-500">
                            Create a new team member profile for your organization.
                        </p>
                    </div>

                    <div className="bg-white shadow overflow-hidden sm:rounded-lg">
                        <div className="px-4 py-5 sm:px-6 border-b border-gray-200">
                            <h3 className="text-lg leading-6 font-medium text-gray-900">Member Information</h3>
                            <p className="mt-1 max-w-2xl text-sm text-gray-500">
                                Provide the personal and professional details for this team member.
                            </p>
                        </div>
                        
                        <form onSubmit={submit} className="space-y-6 p-6">
                            <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                                <div className="sm:col-span-1">
                                    <InputLabel htmlFor="name" value="Full Name" />
                                    <TextInput
                                        id="name"
                                        type="text"
                                        name="name"
                                        value={data.name}
                                        className="mt-1 block w-full"
                                        placeholder="e.g., John Smith"
                                        isFocused={true}
                                        onChange={(e) => setData('name', e.target.value)}
                                    />
                                    <p className="mt-1 text-sm text-gray-500">First and last name</p>
                                    <InputError message={errors.name} className="mt-2" />
                                </div>

                                <div className="sm:col-span-1">
                                    <InputLabel htmlFor="position" value="Job Position" />
                                    <TextInput
                                        id="position"
                                        type="text"
                                        name="position"
                                        value={data.position}
                                        className="mt-1 block w-full"
                                        placeholder="e.g., Senior Developer"
                                        onChange={(e) => setData('position', e.target.value)}
                                    />
                                    <p className="mt-1 text-sm text-gray-500">Job title or position</p>
                                    <InputError message={errors.position} className="mt-2" />
                                </div>
                            </div>

                            <div>
                                <InputLabel htmlFor="department" value="Department" />
                                <div className="mt-1">
                                    <select
                                        id="department"
                                        name="department"
                                        value={data.department}
                                        className="block w-full pl-3 pr-10 py-2 text-base border-gray-300 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm rounded-md"
                                        onChange={(e) => setData('department', e.target.value)}
                                    >
                                        <option value="">Select a department</option>
                                        {departmentOptions.map((dept, index) => (
                                            <option key={index} value={dept}>{dept}</option>
                                        ))}
                                    </select>
                                </div>
                                <p className="mt-1 text-sm text-gray-500">Department or team this member belongs to</p>
                                <InputError message={errors.department} className="mt-2" />
                            </div>

                            <div>
                                <InputLabel htmlFor="description" value="Bio / Description" />
                                <textarea
                                    id="description"
                                    name="description"
                                    rows={4}
                                    value={data.description}
                                    className="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500"
                                    placeholder="Brief professional biography..."
                                    onChange={(e) => setData('description', e.target.value)}
                                ></textarea>
                                <p className="mt-1 text-sm text-gray-500">Professional background and expertise</p>
                                <InputError message={errors.description} className="mt-2" />
                            </div>

                            <div>
                                <InputLabel htmlFor="image" value="Profile Photo" />
                                <div className="mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 border-dashed rounded-md">
                                    <div className="space-y-1 text-center">
                                        {imagePreview ? (
                                            <div className="flex flex-col items-center">
                                                <img src={imagePreview} alt="Preview" className="h-32 w-32 rounded-full object-cover mb-4" />
                                                <button
                                                    type="button"
                                                    onClick={() => {
                                                        setData('image', null);
                                                        setImagePreview(null);
                                                    }}
                                                    className="text-sm text-red-600 hover:text-red-800"
                                                >
                                                    Remove photo
                                                </button>
                                            </div>
                                        ) : (
                                            <>
                                                <svg className="mx-auto h-12 w-12 text-gray-400" stroke="currentColor" fill="none" viewBox="0 0 48 48" aria-hidden="true">
                                                    <path d="M28 8H12a4 4 0 00-4 4v20m32-12v8m0 0v8a4 4 0 01-4 4H12a4 4 0 01-4-4v-4m32-4l-3.172-3.172a4 4 0 00-5.656 0L28 28M8 32l9.172-9.172a4 4 0 015.656 0L28 28m0 0l4 4m4-24h8m-4-4v8m-12 4h.02" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
                                                </svg>
                                                <div className="flex text-sm text-gray-600">
                                                    <label htmlFor="image" className="relative cursor-pointer bg-white rounded-md font-medium text-indigo-600 hover:text-indigo-500 focus-within:outline-none focus-within:ring-2 focus-within:ring-offset-2 focus-within:ring-indigo-500">
                                                        <span>Upload a photo</span>
                                                        <input id="image" name="image" type="file" className="sr-only" onChange={handleImageChange} />
                                                    </label>
                                                    <p className="pl-1">or drag and drop</p>
                                                </div>
                                                <p className="text-xs text-gray-500">PNG, JPG up to 10MB</p>
                                            </>
                                        )}
                                    </div>
                                </div>
                                <p className="mt-1 text-sm text-gray-500">Recommended: Square image, at least 300x300 pixels</p>
                                <InputError message={errors.image} className="mt-2" />
                            </div>

                            <div className="space-y-4">
                                <h3 className="text-lg font-medium text-gray-900">Social Media Profiles</h3>
                                <div className="grid grid-cols-1 gap-4 sm:grid-cols-3">
                                    <div>
                                        <InputLabel htmlFor="facebook_link" value="Facebook" />
                                        <div className="mt-1 flex rounded-md shadow-sm">
                                            <span className="inline-flex items-center px-3 rounded-l-md border border-r-0 border-gray-300 bg-gray-50 text-gray-500 text-sm">
                                                <svg className="h-5 w-5 text-blue-600" fill="currentColor" viewBox="0 0 24 24">
                                                    <path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/>
                                                </svg>
                                            </span>
                                            <TextInput
                                                id="facebook_link"
                                                type="text"
                                                name="facebook_link"
                                                value={data.facebook_link}
                                                className="flex-1 min-w-0 block w-full rounded-none rounded-r-md"
                                                placeholder="Profile URL"
                                                onChange={(e) => setData('facebook_link', e.target.value)}
                                            />
                                        </div>
                                        <InputError message={errors.facebook_link} className="mt-2" />
                                    </div>

                                    <div>
                                        <InputLabel htmlFor="twitter_link" value="Twitter" />
                                        <div className="mt-1 flex rounded-md shadow-sm">
                                            <span className="inline-flex items-center px-3 rounded-l-md border border-r-0 border-gray-300 bg-gray-50 text-gray-500 text-sm">
                                                <svg className="h-5 w-5 text-blue-400" fill="currentColor" viewBox="0 0 24 24">
                                                    <path d="M23.953 4.57a10 10 0 01-2.825.775 4.958 4.958 0 002.163-2.723c-.951.555-2.005.959-3.127 1.184a4.92 4.92 0 00-8.384 4.482C7.69 8.095 4.067 6.13 1.64 3.162a4.822 4.822 0 00-.666 2.475c0 1.71.87 3.213 2.188 4.096a4.904 4.904 0 01-2.228-.616v.06a4.923 4.923 0 003.946 4.827 4.996 4.996 0 01-2.212.085 4.936 4.936 0 004.604 3.417 9.867 9.867 0 01-6.102 2.105c-.39 0-.779-.023-1.17-.067a13.995 13.995 0 007.557 2.209c9.053 0 13.998-7.496 13.998-13.985 0-.21 0-.42-.015-.63A9.935 9.935 0 0024 4.59z"/>
                                                </svg>
                                            </span>
                                            <TextInput
                                                id="twitter_link"
                                                type="text"
                                                name="twitter_link"
                                                value={data.twitter_link}
                                                className="flex-1 min-w-0 block w-full rounded-none rounded-r-md"
                                                placeholder="@username"
                                                onChange={(e) => setData('twitter_link', e.target.value)}
                                            />
                                        </div>
                                        <InputError message={errors.twitter_link} className="mt-2" />
                                    </div>

                                    <div>
                                        <InputLabel htmlFor="linkedin_link" value="LinkedIn" />
                                        <div className="mt-1 flex rounded-md shadow-sm">
                                            <span className="inline-flex items-center px-3 rounded-l-md border border-r-0 border-gray-300 bg-gray-50 text-gray-500 text-sm">
                                                <svg className="h-5 w-5 text-blue-700" fill="currentColor" viewBox="0 0 24 24">
                                                    <path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z"/>
                                                </svg>
                                            </span>
                                            <TextInput
                                                id="linkedin_link"
                                                type="text"
                                                name="linkedin_link"
                                                value={data.linkedin_link}
                                                className="flex-1 min-w-0 block w-full rounded-none rounded-r-md"
                                                placeholder="Profile URL"
                                                onChange={(e) => setData('linkedin_link', e.target.value)}
                                            />
                                        </div>
                                        <InputError message={errors.linkedin_link} className="mt-2" />
                                    </div>
                                </div>
                                <p className="text-sm text-gray-500">Optional: Add social media profile links</p>
                            </div>

                            <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                                <div className="sm:col-span-1">
                                    <InputLabel htmlFor="order" value="Display Order" />
                                    <TextInput
                                        id="order"
                                        type="number"
                                        name="order"
                                        value={data.order}
                                        className="mt-1 block w-full"
                                        placeholder="0"
                                        onChange={(e) => setData('order', e.target.value)}
                                    />
                                    <p className="mt-1 text-sm text-gray-500">Lower numbers appear first</p>
                                    <InputError message={errors.order} className="mt-2" />
                                </div>

                                <div className="sm:col-span-1">
                                    <div className="flex items-center h-full pt-6">
                                        <input
                                            id="is_leadership"
                                            name="is_leadership"
                                            type="checkbox"
                                            checked={data.is_leadership}
                                            onChange={(e) => setData('is_leadership', e.target.checked)}
                                            className="h-4 w-4 text-indigo-600 focus:ring-indigo-500 border-gray-300 rounded"
                                        />
                                        <label htmlFor="is_leadership" className="ml-2 block text-sm text-gray-900">
                                            Leadership team member
                                        </label>
                                    </div>
                                </div>
                            </div>

                            <div className="flex justify-end space-x-3 pt-5 border-t border-gray-200">
                                <Link href={route('admin.teams.index')}>
                                    <SecondaryButton type="button">Cancel</SecondaryButton>
                                </Link>
                                <PrimaryButton disabled={processing} className="ml-3">
                                    {processing ? 'Creating...' : 'Create Team Member'}
                                </PrimaryButton>
                            </div>
                        </form>
                    </div>

                    {/* Help Section */}
                    <div className="mt-8 bg-blue-50 overflow-hidden shadow rounded-lg">
                        <div className="px-4 py-5 sm:p-6">
                            <h3 className="text-lg leading-6 font-medium text-blue-800">Team Member Guidelines</h3>
                            <div className="mt-2 max-w-xl text-sm text-blue-700">
                                <ul className="list-disc pl-5 space-y-1">
                                    <li>Use professional, high-quality profile photos</li>
                                    <li>Write concise bios that highlight expertise and experience</li>
                                    <li>Include relevant social media links when available</li>
                                    <li>Assign appropriate departments for better organization</li>
                                    <li>Mark leadership team members for special display</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}