import React, { useState, useEffect, useRef } from 'react';
import { Head, Link } from '@inertiajs/react';
import HomeLayout from '@/Layouts/HomeLayout';
import { motion } from 'framer-motion';

export default function Show({ post, relatedPosts }) {
    const [readingProgress, setReadingProgress] = useState(0);
    const [isBookmarked, setIsBookmarked] = useState(false);
    const [showShareMenu, setShowShareMenu] = useState(false);
    const [readingTime, setReadingTime] = useState(0);
    const [fontSize, setFontSize] = useState('normal');
    const [isDarkMode, setIsDarkMode] = useState(false);
    const [activeSection, setActiveSection] = useState('');
    const [userReaction, setUserReaction] = useState('');
    const [showTableOfContents, setShowTableOfContents] = useState(false);
    const contentRef = useRef(null);

    // Calculate reading time
    useEffect(() => {
        const wordsPerMinute = 200;
        const wordCount = post.content.split(' ').length;
        setReadingTime(Math.ceil(wordCount / wordsPerMinute));
    }, [post.content]);

    // Calculate reading progress
    useEffect(() => {
        const handleScroll = () => {
            const scrollTop = window.scrollY;
            const docHeight = document.documentElement.scrollHeight - window.innerHeight;
            const scrollPercent = (scrollTop / docHeight) * 100;
            setReadingProgress(scrollPercent);
            
            // Update active section for table of contents
            if (contentRef.current) {
                const sections = contentRef.current.querySelectorAll('h2, h3');
                sections.forEach(section => {
                    const sectionTop = section.offsetTop;
                    if (scrollTop >= sectionTop - 100) {
                        setActiveSection(section.id);
                    }
                });
            }
        };

        window.addEventListener('scroll', handleScroll);
        return () => window.removeEventListener('scroll', handleScroll);
    }, []);

    // Get category info
    const getCategoryInfo = (category) => {
        const categories = {
            'Technology': { color: 'bg-blue-100 text-blue-800', icon: '💻', gradient: 'from-blue-400 to-blue-600' },
            'Health': { color: 'bg-green-100 text-green-800', icon: '🏥', gradient: 'from-green-400 to-green-600' },
            'Business': { color: 'bg-purple-100 text-purple-800', icon: '💼', gradient: 'from-purple-400 to-purple-600' },
            'Education': { color: 'bg-yellow-100 text-yellow-800', icon: '📚', gradient: 'from-yellow-400 to-yellow-600' },
            'Lifestyle': { color: 'bg-pink-100 text-pink-800', icon: '🌟', gradient: 'from-pink-400 to-pink-600' },
            'default': { color: 'bg-gray-100 text-gray-800', icon: '📝', gradient: 'from-gray-400 to-gray-600' }
        };
        return categories[category] || categories['default'];
    };

    // Format date
    const formatDate = (dateString) => {
        const options = { year: 'numeric', month: 'long', day: 'numeric' };
        return new Date(dateString).toLocaleDateString(undefined, options);
    };

    // Share functionality
    const sharePost = (platform) => {
        const url = window.location.href;
        const text = `Check out this article: ${post.title}`;
        
        switch(platform) {
            case 'twitter':
                window.open(`https://twitter.com/intent/tweet?text=${encodeURIComponent(text)}&url=${encodeURIComponent(url)}`, '_blank');
                break;
            case 'facebook':
                window.open(`https://www.facebook.com/sharer/sharer.php?u=${encodeURIComponent(url)}`, '_blank');
                break;
            case 'linkedin':
                window.open(`https://www.linkedin.com/sharing/share-offsite/?url=${encodeURIComponent(url)}`, '_blank');
                break;
            case 'copy':
                navigator.clipboard.writeText(url);
                alert('Link copied to clipboard!');
                break;
            default:
                break;
        }
        setShowShareMenu(false);
    };

    // Generate table of contents
    const generateTableOfContents = () => {
        if (!contentRef.current) return [];
        
        const headings = contentRef.current.querySelectorAll('h2, h3');
        return Array.from(headings).map(heading => ({
            id: heading.id,
            text: heading.textContent,
            level: heading.tagName.toLowerCase()
        }));
    };

    // Font size options
    const fontSizeOptions = {
        small: 'text-sm',
        normal: 'text-base',
        large: 'text-lg',
        xlarge: 'text-xl'
    };

    // Add IDs to headings for table of contents
    useEffect(() => {
        if (contentRef.current) {
            const headings = contentRef.current.querySelectorAll('h2, h3');
            headings.forEach((heading, index) => {
                heading.id = `section-${index}`;
            });
        }
    }, []);

    return (
        <HomeLayout>
            <Head title={post.title} />

            {/* Reading Progress Bar */}
            <div className="fixed top-0 left-0 w-full h-1 bg-gray-200 z-50">
                <motion.div 
                    className="h-full bg-gradient-to-r from-emerald-500 to-teal-600"
                    style={{ width: `${readingProgress}%` }}
                />
            </div>

            {/* Floating Action Buttons */}
            <div className="fixed right-8 bottom-8 z-40 flex flex-col space-y-3">
                {/* Table of Contents Toggle */}
                <motion.button
                    onClick={() => setShowTableOfContents(!showTableOfContents)}
                    className="w-14 h-14 bg-white rounded-full shadow-lg flex items-center justify-center text-gray-700 hover:bg-gray-100 transition-colors"
                    whileHover={{ scale: 1.1 }}
                    whileTap={{ scale: 0.9 }}
                >
                    <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                    </svg>
                </motion.button>
                
                {/* Reading Settings */}
                <motion.button
                    className="w-14 h-14 bg-white rounded-full shadow-lg flex items-center justify-center text-gray-700 hover:bg-gray-100 transition-colors"
                    whileHover={{ scale: 1.1 }}
                    whileTap={{ scale: 0.9 }}
                >
                    <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 6V4m0 2a2 2 0 100 4m0-4a2 2 0 110 4m-6 8a2 2 0 100-4m0 4a2 2 0 110-4m0 4v2m0-6V4m6 6v10m6-2a2 2 0 100-4m0 4a2 2 0 110-4m0 4v2m0-6V4" />
                    </svg>
                </motion.button>
                
                {/* Scroll to Top */}
                <motion.button
                    onClick={() => window.scrollTo({ top: 0, behavior: 'smooth' })}
                    className="w-14 h-14 bg-gradient-to-r from-emerald-500 to-teal-600 rounded-full shadow-lg flex items-center justify-center text-white"
                    whileHover={{ scale: 1.1 }}
                    whileTap={{ scale: 0.9 }}
                >
                    <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 10l7-7m0 0l7 7m-7-7v18" />
                    </svg>
                </motion.button>
            </div>

            {/* Table of Contents Sidebar */}
            {showTableOfContents && (
                <motion.div 
                    className="fixed right-0 top-20 bottom-0 w-80 bg-white shadow-2xl z-30 overflow-y-auto p-6"
                    initial={{ x: '100%' }}
                    animate={{ x: 0 }}
                    exit={{ x: '100%' }}
                    transition={{ type: 'spring', damping: 25, stiffness: 300 }}
                >
                    <div className="flex justify-between items-center mb-6">
                        <h3 className="text-lg font-bold text-gray-900">Table of Contents</h3>
                        <button
                            onClick={() => setShowTableOfContents(false)}
                            className="text-gray-500 hover:text-gray-700"
                        >
                            <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                            </svg>
                        </button>
                    </div>
                    
                    <div className="space-y-2">
                        {generateTableOfContents().map((item, index) => (
                            <a
                                key={index}
                                href={`#${item.id}`}
                                className={`block py-2 px-3 rounded-md text-sm ${
                                    activeSection === item.id
                                        ? 'bg-emerald-100 text-emerald-700 font-medium'
                                        : 'text-gray-700 hover:bg-gray-100'
                                } ${item.level === 'h3' ? 'ml-4' : ''}`}
                            >
                                {item.text}
                            </a>
                        ))}
                    </div>
                </motion.div>
            )}

            {/* Article Header with Hero Image */}
            <div className="relative">
                <div className="relative h-96 md:h-[500px] overflow-hidden">
                    <img 
                        src={post.image_url} 
                        alt={post.title} 
                        className="w-full h-full object-cover"
                    />
                    <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/40 to-transparent"></div>
                    
                    {/* Decorative Elements */}
                    <div className="absolute inset-0 overflow-hidden pointer-events-none">
                        <div className="absolute top-20 left-10 w-32 h-32 bg-emerald-500/20 rounded-full blur-2xl"></div>
                        <div className="absolute bottom-20 right-10 w-40 h-40 bg-teal-500/20 rounded-full blur-2xl"></div>
                    </div>
                    
                    {/* Back to Blog Button */}
                    <motion.div 
                        className="absolute top-8 left-8 z-10"
                        initial={{ opacity: 0, x: -20 }}
                        animate={{ opacity: 1, x: 0 }}
                        transition={{ delay: 0.3 }}
                    >
                        <Link 
                            href={route('blog.index')} 
                            className="flex items-center text-white bg-white/20 backdrop-blur-md px-4 py-2 rounded-full hover:bg-white/30 transition-colors"
                        >
                            <svg className="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 19l-7-7m0 0l7-7m-7 7h18" />
                            </svg>
                            Back to Blog
                        </Link>
                    </motion.div>
                    
                    {/* Article Title and Meta */}
                    <div className="absolute bottom-0 left-0 right-0 p-8 md:p-12">
                        <motion.div
                            initial={{ opacity: 0, y: 30 }}
                            animate={{ opacity: 1, y: 0 }}
                            transition={{ delay: 0.2 }}
                        >
                            <div className="max-w-4xl mx-auto">
                                <div className="flex items-center mb-4">
                                    <span className={`inline-flex items-center px-3 py-1 text-sm font-semibold rounded-full ${getCategoryInfo(post.category || 'default').color}`}>
                                        <span className="mr-1">{getCategoryInfo(post.category || 'default').icon}</span>
                                        {post.category || 'Article'}
                                    </span>
                                    <span className="ml-3 text-white/80 text-sm flex items-center">
                                        <svg className="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                                        </svg>
                                        {readingTime} min read
                                    </span>
                                </div>
                                <h1 className="text-3xl md:text-5xl font-bold text-white mb-4 leading-tight">{post.title}</h1>
                                <div className="flex items-center text-white/90">
                                    <div className="w-12 h-12 bg-gradient-to-br from-emerald-400 to-teal-600 rounded-full mr-4 flex items-center justify-center text-white font-bold">
                                        {(post.user ? post.user.name : 'Unknown Author')[0].toUpperCase()}
                                    </div>
                                    <div>
                                        <div className="font-medium">{post.user ? post.user.name : 'Unknown Author'}</div>
                                        <div className="text-sm text-white/70">
                                            {formatDate(post.created_at)}
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </motion.div>
                    </div>
                </div>
            </div>

            {/* Reading Settings Bar */}
            <div className="sticky top-0 bg-white border-b border-gray-200 z-20 py-3">
                <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 flex justify-between items-center">
                    <div className="flex items-center space-x-4">
                        <span className="text-sm text-gray-600">Reading Progress:</span>
                        <div className="w-32 h-2 bg-gray-200 rounded-full overflow-hidden">
                            <div 
                                className="h-full bg-gradient-to-r from-emerald-500 to-teal-600"
                                style={{ width: `${readingProgress}%` }}
                            ></div>
                        </div>
                        <span className="text-sm text-gray-600">{Math.round(readingProgress)}%</span>
                    </div>
                    
                    <div className="flex items-center space-x-4">
                        {/* Font Size Controls */}
                        <div className="flex items-center space-x-2">
                            <span className="text-sm text-gray-600">Font Size:</span>
                            <button
                                onClick={() => setFontSize('small')}
                                className={`px-2 py-1 text-xs rounded ${fontSize === 'small' ? 'bg-emerald-100 text-emerald-700' : 'text-gray-600 hover:bg-gray-100'}`}
                            >
                                A-
                            </button>
                            <button
                                onClick={() => setFontSize('normal')}
                                className={`px-2 py-1 text-sm rounded ${fontSize === 'normal' ? 'bg-emerald-100 text-emerald-700' : 'text-gray-600 hover:bg-gray-100'}`}
                            >
                                A
                            </button>
                            <button
                                onClick={() => setFontSize('large')}
                                className={`px-2 py-1 text-base rounded ${fontSize === 'large' ? 'bg-emerald-100 text-emerald-700' : 'text-gray-600 hover:bg-gray-100'}`}
                            >
                                A+
                            </button>
                            <button
                                onClick={() => setFontSize('xlarge')}
                                className={`px-2 py-1 text-lg rounded ${fontSize === 'xlarge' ? 'bg-emerald-100 text-emerald-700' : 'text-gray-600 hover:bg-gray-100'}`}
                            >
                                A++
                            </button>
                        </div>
                        
                        {/* Dark Mode Toggle */}
                        <button
                            onClick={() => setIsDarkMode(!isDarkMode)}
                            className="p-2 rounded-full text-gray-600 hover:bg-gray-100"
                        >
                            {isDarkMode ? (
                                <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 3v1m0 16v1m9-9h-1M4 12H3m15.364 6.364l-.707-.707M6.343 6.343l-.707-.707m12.728 0l-.707.707M6.343 17.657l-.707.707M16 12a4 4 0 11-8 0 4 4 0 018 0z" />
                                </svg>
                            ) : (
                                <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9.003 9.003 0 008.354-5.646z" />
                                </svg>
                            )}
                        </button>
                    </div>
                </div>
            </div>

            {/* Article Content */}
            <section className={`py-12 ${isDarkMode ? 'bg-gray-900 text-white' : 'bg-white'}`}>
                <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Article Actions */}
                    <motion.div 
                        className={`flex justify-between items-center mb-8 pb-6 ${isDarkMode ? 'border-gray-800' : 'border-gray-200'}`}
                        initial={{ opacity: 0 }}
                        whileInView={{ opacity: 1 }}
                        viewport={{ once: true }}
                    >
                        <div className="flex items-center space-x-4">
                            <button
                                onClick={() => setIsBookmarked(!isBookmarked)}
                                className={`flex items-center ${isDarkMode ? 'text-gray-400 hover:text-emerald-400' : 'text-gray-600 hover:text-emerald-600'} transition-colors`}
                            >
                                <svg className={`w-6 h-6 mr-2 ${isBookmarked ? 'fill-current text-emerald-600' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                                </svg>
                                {isBookmarked ? 'Bookmarked' : 'Bookmark'}
                            </button>
                            
                            <div className="relative">
                                <button
                                    onClick={() => setShowShareMenu(!showShareMenu)}
                                    className={`flex items-center ${isDarkMode ? 'text-gray-400 hover:text-emerald-400' : 'text-gray-600 hover:text-emerald-600'} transition-colors`}
                                >
                                    <svg className="w-6 h-6 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8.684 13.342C8.886 12.938 9 12.482 9 12c0-.482-.114-.938-.316-1.342m0 2.684a3 3 0 110-2.684m9.032 4.026a9.001 9.001 0 01-7.432 0m9.032-4.026A9.001 9.001 0 0112 3c-4.474 0-8.268 3.12-9.032 7.326m0 0A9.001 9.001 0 0012 21c4.474 0 8.268-3.12 9.032-7.326" />
                                    </svg>
                                    Share
                                </button>
                                
                                {showShareMenu && (
                                    <motion.div 
                                        className={`absolute top-12 right-0 ${isDarkMode ? 'bg-gray-800' : 'bg-white'} rounded-lg shadow-xl p-2 z-10 min-w-[200px]`}
                                        initial={{ opacity: 0, y: -10 }}
                                        animate={{ opacity: 1, y: 0 }}
                                    >
                                        <button
                                            onClick={() => sharePost('twitter')}
                                            className={`w-full text-left px-4 py-2 ${isDarkMode ? 'hover:bg-gray-700' : 'hover:bg-gray-100'} rounded flex items-center`}
                                        >
                                            <svg className="w-5 h-5 mr-3 text-blue-400" fill="currentColor" viewBox="0 0 24 24">
                                                <path d="M23.953 4.57a10 10 0 01-2.825.775 4.958 4.958 0 002.163-2.723c-.951.555-2.005.959-3.127 1.184a4.92 4.92 0 00-8.384 4.482C7.69 8.095 4.067 6.13 1.64 3.162a4.822 4.822 0 00-.666 2.475c0 1.71.87 3.213 2.188 4.096a4.904 4.904 0 01-2.228-.616v.06a4.923 4.923 0 003.946 4.827 4.996 4.996 0 01-2.212.085 4.936 4.936 0 004.604 3.417 9.867 9.867 0 01-6.102 2.105c-.39 0-.779-.023-1.17-.067a13.995 13.995 0 007.557 2.209c9.053 0 13.998-7.496 13.998-13.985 0-.21 0-.42-.015-.63A9.935 9.935 0 0024 4.59z"/>
                                            </svg>
                                            Twitter
                                        </button>
                                        <button
                                            onClick={() => sharePost('facebook')}
                                            className={`w-full text-left px-4 py-2 ${isDarkMode ? 'hover:bg-gray-700' : 'hover:bg-gray-100'} rounded flex items-center`}
                                        >
                                            <svg className="w-5 h-5 mr-3 text-blue-600" fill="currentColor" viewBox="0 0 24 24">
                                                <path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/>
                                            </svg>
                                            Facebook
                                        </button>
                                        <button
                                            onClick={() => sharePost('linkedin')}
                                            className={`w-full text-left px-4 py-2 ${isDarkMode ? 'hover:bg-gray-700' : 'hover:bg-gray-100'} rounded flex items-center`}
                                        >
                                            <svg className="w-5 h-5 mr-3 text-blue-700" fill="currentColor" viewBox="0 0 24 24">
                                                <path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z"/>
                                            </svg>
                                            LinkedIn
                                        </button>
                                        <button
                                            onClick={() => sharePost('copy')}
                                            className={`w-full text-left px-4 py-2 ${isDarkMode ? 'hover:bg-gray-700' : 'hover:bg-gray-100'} rounded flex items-center`}
                                        >
                                            <svg className="w-5 h-5 mr-3 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 5H6a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2v-1M8 5a2 2 0 002 2h2a2 2 0 002-2M8 5a2 2 0 012-2h2a2 2 0 012 2m0 0h2a2 2 0 012 2v3m2 4H10m0 0l3-3m-3 3l3 3" />
                                            </svg>
                                            Copy Link
                                        </button>
                                    </motion.div>
                                )}
                            </div>
                        </div>
                        
                        <div className={`flex items-center ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>
                            <svg className="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                            </svg>
                            <span>1,234 views</span>
                        </div>
                    </motion.div>

                    {/* Article Body with Enhanced Styling */}
                    <motion.div 
                        ref={contentRef}
                        className={`prose prose-lg max-w-none mb-12 ${fontSizeOptions[fontSize]} ${isDarkMode ? 'prose-invert' : ''}`}
                        initial={{ opacity: 0 }}
                        whileInView={{ opacity: 1 }}
                        viewport={{ once: true }}
                    >
                        {/* Pull Quote */}
                        <div className={`my-12 p-6 ${isDarkMode ? 'bg-gray-800' : 'bg-emerald-50'} rounded-2xl border-l-4 border-emerald-500`}>
                            <svg className={`w-8 h-8 ${isDarkMode ? 'text-emerald-400' : 'text-emerald-600'} mb-4`} fill="currentColor" viewBox="0 0 24 24">
                                <path d="M14.017 21v-7.391c0-5.704 3.731-9.57 8.983-10.609l.995 2.151c-2.432.917-3.995 3.638-3.995 5.849h4v10h-9.983zm-14.017 0v-7.391c0-5.704 3.748-9.57 9-10.609l.996 2.151c-2.433.917-3.996 3.638-3.996 5.849h3.983v10h-9.983z" />
                            </svg>
                            <p className={`text-xl ${isDarkMode ? 'text-gray-200' : 'text-gray-700'} font-medium italic`}>
                                "The key to successful blogging is to provide value to your readers. Every post should teach something, inspire someone, or solve a problem."
                            </p>
                        </div>

                        {/* Content with decorative elements */}
                        <div dangerouslySetInnerHTML={{ __html: post.content }} />
                        
                        {/* Decorative Divider */}
                        <div className="my-12 flex items-center justify-center">
                            <div className={`h-px ${isDarkMode ? 'bg-gray-700' : 'bg-gray-300'} flex-1`}></div>
                            <div className="px-4">
                                <div className="w-8 h-8 bg-gradient-to-br from-emerald-400 to-teal-600 rounded-full flex items-center justify-center">
                                    <svg className="w-4 h-4 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                                    </svg>
                                </div>
                            </div>
                            <div className={`h-px ${isDarkMode ? 'bg-gray-700' : 'bg-gray-300'} flex-1`}></div>
                        </div>
                    </motion.div>

                    {/* Article Reactions */}
                    <motion.div 
                        className={`mb-12 p-6 rounded-2xl ${isDarkMode ? 'bg-gray-800' : 'bg-gray-50'}`}
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        viewport={{ once: true }}
                    >
                        <h3 className="text-lg font-semibold mb-4">How did you find this article?</h3>
                        <div className="flex space-x-4">
                            {[
                                { emoji: '👍', label: 'Helpful' },
                                { emoji: '❤️', label: 'Love it' },
                                { emoji: '🤔', label: 'Thought-provoking' },
                                { emoji: '😄', label: 'Funny' },
                                { emoji: '👎', label: 'Not helpful' }
                            ].map((reaction) => (
                                <button
                                    key={reaction.label}
                                    onClick={() => setUserReaction(reaction.label)}
                                    className={`flex flex-col items-center p-3 rounded-lg transition-all ${
                                        userReaction === reaction.label
                                            ? 'bg-emerald-100 text-emerald-700'
                                            : isDarkMode
                                                ? 'bg-gray-700 text-gray-300 hover:bg-gray-600'
                                                : 'bg-white text-gray-700 hover:bg-gray-100'
                                    }`}
                                >
                                    <span className="text-2xl mb-1">{reaction.emoji}</span>
                                    <span className="text-xs">{reaction.label}</span>
                                </button>
                            ))}
                        </div>
                    </motion.div>

                    {/* Article Tags */}
                    <motion.div 
                        className={`mb-12 pb-12 ${isDarkMode ? 'border-gray-800' : 'border-gray-200'}`}
                        initial={{ opacity: 0 }}
                        whileInView={{ opacity: 1 }}
                        viewport={{ once: true }}
                    >
                        <h3 className="text-lg font-semibold mb-4">Tags</h3>
                        <div className="flex flex-wrap gap-2">
                            {['Web Development', 'React', 'Laravel', 'JavaScript', 'Design'].map((tag, index) => (
                                <span 
                                    key={index} 
                                    className={`px-3 py-1 ${isDarkMode ? 'bg-gray-800 text-gray-300 hover:bg-emerald-900' : 'bg-gray-100 text-gray-700 hover:bg-emerald-100'} rounded-full text-sm transition-colors cursor-pointer`}
                                >
                                    #{tag}
                                </span>
                            ))}
                        </div>
                    </motion.div>

                    {/* Author Bio */}
                    <motion.div 
                        className={`${isDarkMode ? 'bg-gray-800' : 'bg-gray-50'} rounded-2xl p-8 mb-12`}
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        viewport={{ once: true }}
                    >
                        <div className="flex flex-col md:flex-row items-center md:items-start">
                            <div className={`w-20 h-20 bg-gradient-to-br ${getCategoryInfo(post.category || 'default').gradient} rounded-full mr-6 flex items-center justify-center text-white text-2xl font-bold mb-4 md:mb-0`}>
                                {(post.user ? post.user.name : 'Unknown Author')[0].toUpperCase()}
                            </div>
                            <div className="text-center md:text-left flex-1">
                                <h3 className="text-xl font-bold mb-2">{post.user ? post.user.name : 'Unknown Author'}</h3>
                                <p className={`${isDarkMode ? 'text-gray-400' : 'text-gray-600'} mb-4`}>
                                    Passionate writer and tech enthusiast with over 5 years of experience in web development and design. 
                                    Love sharing knowledge and helping others learn.
                                </p>
                                <div className="flex justify-center md:justify-start space-x-4">
                                    <a href="#" className={`${isDarkMode ? 'text-gray-400 hover:text-emerald-400' : 'text-gray-400 hover:text-emerald-600'} transition-colors`}>
                                        <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                                            <path d="M23.953 4.57a10 10 0 01-2.825.775 4.958 4.958 0 002.163-2.723c-.951.555-2.005.959-3.127 1.184a4.92 4.92 0 00-8.384 4.482C7.69 8.095 4.067 6.13 1.64 3.162a4.822 4.822 0 00-.666 2.475c0 1.71.87 3.213 2.188 4.096a4.904 4.904 0 01-2.228-.616v.06a4.923 4.923 0 003.946 4.827 4.996 4.996 0 01-2.212.085 4.936 4.936 0 004.604 3.417 9.867 9.867 0 01-6.102 2.105c-.39 0-.779-.023-1.17-.067a13.995 13.995 0 007.557 2.209c9.053 0 13.998-7.496 13.998-13.985 0-.21 0-.42-.015-.63A9.935 9.935 0 0024 4.59z"/>
                                        </svg>
                                    </a>
                                    <a href="#" className={`${isDarkMode ? 'text-gray-400 hover:text-emerald-400' : 'text-gray-400 hover:text-emerald-600'} transition-colors`}>
                                        <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                                            <path d="M12 0c-6.626 0-12 5.373-12 12 0 5.302 3.438 9.8 8.207 11.387.599.111.793-.261.793-.577v-2.234c-3.338.726-4.033-1.416-4.033-1.416-.546-1.387-1.333-1.756-1.333-1.756-1.089-.745.083-.729.083-.729 1.205.084 1.839 1.237 1.839 1.237 1.07 1.834 2.807 1.304 3.492.997.107-.775.418-1.305.762-1.604-2.665-.305-5.467-1.334-5.467-5.931 0-1.311.469-2.381 1.236-3.221-.124-.303-.535-1.524.117-3.176 0 0 1.008-.322 3.301 1.23.957-.266 1.983-.399 3.003-.404 1.02.005 2.047.138 3.006.404 2.291-1.552 3.297-1.23 3.297-1.23.653 1.653.242 2.874.118 3.176.77.84 1.235 1.911 1.235 3.221 0 4.609-2.807 5.624-5.479 5.921.43.372.823 1.102.823 2.222v3.293c0 .319.192.694.801.576 4.765-1.589 8.199-6.086 8.199-11.386 0-6.627-5.373-12-12-12z"/>
                                        </svg>
                                    </a>
                                    <a href="#" className={`${isDarkMode ? 'text-gray-400 hover:text-emerald-400' : 'text-gray-400 hover:text-emerald-600'} transition-colors`}>
                                        <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                                            <path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z"/>
                                        </svg>
                                    </a>
                                </div>
                            </div>
                        </div>
                    </motion.div>

                    {/* Comments Section */}
                    <motion.div 
                        className="mb-12"
                        initial={{ opacity: 0 }}
                        whileInView={{ opacity: 1 }}
                        viewport={{ once: true }}
                    >
                        <h3 className="text-2xl font-bold mb-6">Comments (3)</h3>
                        
                        {/* Comment Form */}
                        <div className={`${isDarkMode ? 'bg-gray-800' : 'bg-gray-50'} rounded-2xl p-6 mb-8`}>
                            <h4 className="text-lg font-semibold mb-4">Leave a Comment</h4>
                            <form>
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                                    <input
                                        type="text"
                                        placeholder="Your Name"
                                        className={`px-4 py-3 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'} rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500`}
                                    />
                                    <input
                                        type="email"
                                        placeholder="Your Email"
                                        className={`px-4 py-3 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'} rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500`}
                                    />
                                </div>
                                <textarea
                                    placeholder="Your Comment"
                                    rows="4"
                                    className={`w-full px-4 py-3 ${isDarkMode ? 'bg-gray-700 border-gray-600 text-white' : 'bg-white border-gray-300'} rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500 mb-4`}
                                ></textarea>
                                <button
                                    type="submit"
                                    className="px-6 py-3 bg-gradient-to-r from-emerald-500 to-teal-600 text-white font-medium rounded-lg hover:from-emerald-600 hover:to-teal-700 transition-all shadow-md"
                                >
                                    Post Comment
                                </button>
                            </form>
                        </div>
                        
                        {/* Comments List */}
                        <div className="space-y-6">
                            {[1, 2, 3].map((comment) => (
                                <div key={comment} className="flex space-x-4">
                                    <div className="w-12 h-12 bg-gradient-to-br from-emerald-400 to-teal-600 rounded-full flex-shrink-0 flex items-center justify-center text-white font-bold">
                                        JD
                                    </div>
                                    <div className="flex-1">
                                        <div className={`${isDarkMode ? 'bg-gray-800' : 'bg-gray-50'} rounded-2xl p-6`}>
                                            <div className="flex items-center justify-between mb-2">
                                                <h4 className="font-semibold">John Doe</h4>
                                                <span className={`text-sm ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`}>2 days ago</span>
                                            </div>
                                            <p className={isDarkMode ? 'text-gray-300' : 'text-gray-700'}>
                                                This is a great article! I really enjoyed reading it and learned a lot. 
                                                Keep up the excellent work.
                                            </p>
                                        </div>
                                        <button className={`mt-2 text-sm ${isDarkMode ? 'text-gray-400 hover:text-emerald-400' : 'text-gray-500 hover:text-emerald-600'} transition-colors`}>
                                            Reply
                                        </button>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </motion.div>
                </div>
            </section>

            {/* Related Posts Section */}
            {relatedPosts && relatedPosts.length > 0 && (
                <section className={`py-16 ${isDarkMode ? 'bg-gray-900' : 'bg-gray-50'}`}>
                    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                        <motion.div 
                            className="text-center mb-12"
                            initial={{ opacity: 0 }}
                            whileInView={{ opacity: 1 }}
                            viewport={{ once: true }}
                        >
                            <h2 className="text-3xl font-bold mb-4">Related Articles</h2>
                            <div className="w-24 h-1 bg-gradient-to-r from-emerald-500 to-teal-600 mx-auto"></div>
                        </motion.div>
                        
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                            {relatedPosts.map((relatedPost, index) => (
                                <motion.div 
                                    key={relatedPost.id} 
                                    className={`${isDarkMode ? 'bg-gray-800' : 'bg-white'} rounded-xl shadow-md overflow-hidden hover:shadow-xl transition-shadow duration-300 group`}
                                    initial={{ opacity: 0, y: 30 }}
                                    whileInView={{ opacity: 1, y: 0 }}
                                    viewport={{ once: true }}
                                    transition={{ delay: index * 0.1 }}
                                >
                                    <div className="relative h-48 overflow-hidden">
                                        <img 
                                            src={relatedPost.image_url} 
                                            alt={relatedPost.title} 
                                            className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-500"
                                        />
                                        <div className="absolute top-4 left-4">
                                            <span className={`inline-block px-3 py-1 text-xs font-semibold rounded-full ${getCategoryInfo(relatedPost.category || 'default').color}`}>
                                                {relatedPost.category || 'Article'}
                                            </span>
                                        </div>
                                    </div>
                                    <div className="p-6">
                                        <h3 className={`text-xl font-bold mb-3 group-hover:text-emerald-600 transition-colors ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>
                                            {relatedPost.title}
                                        </h3>
                                        <p className={`mb-4 line-clamp-3 ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>
                                            {relatedPost.content.substring(0, 100)}...
                                        </p>
                                        <Link 
                                            href={route('blog.show', relatedPost.id)} 
                                            className="inline-flex items-center text-emerald-600 font-medium hover:text-emerald-700 transition-colors"
                                        >
                                            Read More
                                            <svg className="w-4 h-4 ml-1 group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M14 5l7 7m0 0l-7 7m7-7H3" />
                                            </svg>
                                        </Link>
                                    </div>
                                </motion.div>
                            ))}
                        </div>
                    </div>
                </section>
            )}

            {/* Newsletter Section */}
            <section className="py-16 bg-gradient-to-br from-emerald-500 via-teal-600 to-cyan-700 relative overflow-hidden">
                {/* Decorative Elements */}
                <div className="absolute inset-0 overflow-hidden">
                    <motion.div 
                        className="absolute top-10 left-10 w-32 h-32 bg-white/10 rounded-full"
                        animate={{ 
                            scale: [1, 1.2, 1],
                            opacity: [0.3, 0.6, 0.3],
                        }}
                        transition={{ 
                            duration: 4,
                            repeat: Infinity,
                            repeatType: "reverse"
                        }}
                    />
                    <motion.div 
                        className="absolute bottom-10 right-10 w-40 h-40 bg-white/10 rounded-full"
                        animate={{ 
                            scale: [1.2, 1, 1.2],
                            opacity: [0.6, 0.3, 0.6],
                        }}
                        transition={{ 
                            duration: 5,
                            repeat: Infinity,
                            repeatType: "reverse"
                        }}
                    />
                </div>
                
                <div className="relative max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        viewport={{ once: true }}
                    >
                        <h2 className="text-3xl font-bold text-white mb-4">
                            Enjoyed this article?
                        </h2>
                        <p className="text-xl text-white/90 mb-8">
                            Subscribe to our newsletter to get more great content delivered to your inbox
                        </p>
                        
                        <form className="flex flex-col sm:flex-row max-w-md mx-auto gap-4">
                            <input
                                type="email"
                                placeholder="Enter your email address"
                                className="flex-1 px-6 py-4 rounded-lg focus:outline-none focus:ring-4 focus:ring-white/30 text-gray-900"
                                required
                            />
                            <motion.button
                                type="submit"
                                className="px-8 py-4 bg-gray-900 text-white font-semibold rounded-lg hover:bg-gray-800 transition-colors shadow-lg"
                                whileHover={{ scale: 1.05 }}
                                whileTap={{ scale: 0.95 }}
                            >
                                Subscribe
                            </motion.button>
                        </form>
                    </motion.div>
                </div>
            </section>
        </HomeLayout>
    );
}