import { useEffect, useState } from 'react';
import { useForm, router } from '@inertiajs/react';
import axios from 'axios';
import { showErrorToast, showSuccessToast } from '@/Components/ToastNote';
import Swal from 'sweetalert2';
import { FaArrowLeft, FaCreditCard, FaHeart, FaLock, FaMinus, FaPlus, FaShoppingBag, FaTag, FaTrash, FaTruck, FaUndo, FaCcPaypal } from 'react-icons/fa';
import HomeLayout from '@/Layouts/HomeLayout';
import { motion } from 'framer-motion';
import { AnimatePresence } from 'framer-motion';
import { XMarkIcon } from '@heroicons/react/24/outline';

export default function View({ cartItems, cartTotal }) {
    const { patch, delete: destroy } = useForm();
    const [couponCode, setCouponCode] = useState('');
    const [discount, setDiscount] = useState(0);
    const [formData, setFormData] = useState({
        name: '',
        email: '',
        address: '',
    });
    const [paymentMethod, setPaymentMethod] = useState('paystack');
    const [isProcessing, setIsProcessing] = useState(false);
    const [paystackAuthUrl, setPaystackAuthUrl] = useState(null);
    const [isIframeLoading, setIsIframeLoading] = useState(true);
    const [paymentCompleted, setPaymentCompleted] = useState(false);
    const [showSuccessModalContent, setShowSuccessModalContent] = useState(false);

    const handleInputChange = (e) => {
        const { name, value } = e.target;
        setFormData(prev => ({ ...prev, [name]: value }));
    };

    const isFormValid = () => {
        return formData.name && formData.email && formData.address && paymentMethod;
    };

    const getSubtotal = () => {
        return Object.values(cartItems).reduce((total, item) => total + item.price * item.quantity, 0);
    };

    const getTotalWithDiscountAndTax = () => {
        const subtotal = getSubtotal();
        const totalAfterDiscount = subtotal - discount;
        const tax = totalAfterDiscount * 0.08; // Assuming 8% tax
        return (totalAfterDiscount + tax).toFixed(2);
    };

    // New useEffect for postMessage listener
    useEffect(() => {
        const handleMessage = (event) => {
            try {
                const message = JSON.parse(event.data);
                if (message.status === 'success') {
                    handlePaymentSuccess(message.data);
                }
            } catch (error) {
                console.error('Error parsing postMessage data:', error);
            }
        };

        window.addEventListener('message', handleMessage);

        return () => {
            window.removeEventListener('message', handleMessage);
        };
    }, []); // Empty dependency array means this runs once on mount

    const handlePaymentSuccess = (data) => {
        console.log("Payment data from iframe:", data);
        setPaystackAuthUrl(null); // Hide the iframe
        setIsIframeLoading(true);
        setPaymentCompleted(true); // Mark payment as completed
        setShowSuccessModalContent(true); // Show success content in modal

        showSuccessToast('Payment successful! Your order is being processed.');
        router.reload({ only: ['cartItems', 'cartTotal'] }); // Reload cart data
    };

    const handleClosePaymentIframe = () => {
        if (showSuccessModalContent) {
            // If success content is showing, just close and reset
            setPaystackAuthUrl(null);
            setIsIframeLoading(true);
            setPaymentCompleted(false);
            setShowSuccessModalContent(false);
        } else {
            // If payment was cancelled
            setPaystackAuthUrl(null);
            setIsIframeLoading(true);

            Swal.fire({
                title: 'Payment Cancelled',
                text: 'Your payment was not completed. You can try again or select a different payment method.',
                icon: 'info',
                confirmButtonText: 'OK',
                confirmButtonColor: '#059669',
            });
        }
    };

    const handleCheckout = async () => {
        if (!isFormValid()) return;
        setIsProcessing(true);

        const checkoutData = {
            ...formData,
            cart: Object.values(cartItems).map(item => ({
                id: item.id,
                name: item.name,
                quantity: item.quantity,
                price: item.price,
            })),
            total: getTotalWithDiscountAndTax(),
            payment_method: paymentMethod,
        };

        try {
            const response = await axios.post(route('checkout.store'), checkoutData);
            
            if (response.data && response.data.authorization_url) {
                if (paymentMethod === 'paystack') {
                    setPaystackAuthUrl(response.data.authorization_url);
                } else if (paymentMethod === 'paypal') {
                    window.location.href = response.data.authorization_url;
                }
            } else {
                showErrorToast('Failed to retrieve payment URL. Please try again.');
                console.error('Payment initiation response is missing authorization_url', response);
            }
        } catch (error) {
            console.error('Checkout error:', error);
            const errorMessage = error.response?.data?.message || 'An unknown error occurred during checkout.';
            showErrorToast(errorMessage);
        } finally {
            setIsProcessing(false);
        }
    };

    const updateQuantity = (rowId, newQuantity) => {
        if (newQuantity <= 0) {
            handleRemoveItem(rowId);
            return;
        }
        
        patch(route('cart.update', rowId), {
            quantity: newQuantity
        });
    };

    const handleRemoveItem = (rowId) => {
        destroy(route('cart.remove', rowId));
    };

    const applyCoupon = () => {
        // This would typically make an API call to validate the coupon
        // For demo purposes, we'll just apply a 10% discount
        if (couponCode.toLowerCase() === 'save10') {
            setDiscount(getSubtotal() * 0.1);
        } else {
            // Show error message
            showErrorToast('Invalid coupon code');
        }
    };

    // Helper function to get the first image from the array
    const getFirstImage = (item) => {
        // Check if image_urls exists and is an array with at least one element
        if (item.attributes && item.attributes.image_urls && Array.isArray(item.attributes.image_urls) && item.attributes.image_urls.length > 0) {
            return item.attributes.image_urls[0];
        }
        // Default placeholder
        return '/images/product-placeholder.jpg';
    };

    return (
        <HomeLayout>
            <AnimatePresence>
                {paystackAuthUrl && (
                    <motion.div
                        className="fixed inset-0 bg-gray-900 bg-opacity-60 flex items-center justify-center z-50 p-2 sm:p-4"
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        exit={{ opacity: 0 }}
                        transition={{ duration: 0.3 }}
                    >
                        <motion.div
                            className="bg-white rounded-xl sm:rounded-2xl shadow-2xl max-w-md w-full max-h-[90vh] overflow-hidden transform"
                            initial={{ scale: 0.9, opacity: 0 }}
                            animate={{ scale: 1, opacity: 1 }}
                            exit={{ scale: 0.9, opacity: 0 }}
                            transition={{ duration: 0.3, ease: "easeOut" }}
                        >
                            <div className="relative bg-gradient-to-r from-emerald-500 to-teal-600 p-3 sm:p-4 md:p-6 text-white">
                                <button
                                    onClick={handleClosePaymentIframe}
                                    className="absolute top-2 sm:top-3 md:top-4 right-2 sm:right-3 md:right-4 text-white/80 hover:text-white transition-colors p-1"
                                    aria-label="Close payment modal"
                                >
                                    <XMarkIcon className="h-4 w-4 sm:h-5 sm:w-5 md:h-6 md:w-6" />
                                </button>
                                <h2 className="text-base sm:text-lg md:text-2xl font-bold">{showSuccessModalContent ? 'Payment Successful!' : 'Complete Your Payment'}</h2>
                                <p className="text-emerald-100 text-xs sm:text-sm mt-1">{showSuccessModalContent ? 'Your order has been placed.' : 'You are securely redirected to our payment partner.'}</p>
                            </div>

                            <div className="p-3 sm:p-4 md:p-6 overflow-y-auto max-h-[60vh]">
                                {showSuccessModalContent ? (
                                    <div className="text-center py-10">
                                        <svg className="mx-auto h-16 w-16 text-emerald-500" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                                        <h3 className="mt-4 text-lg font-medium text-gray-900">Thank You for Your Order!</h3>
                                        <p className="mt-2 text-sm text-gray-500">Your payment was successful and your order is being processed.</p>
                                        <div className="mt-6">
                                            <button
                                                onClick={handleClosePaymentIframe}
                                                className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-emerald-600 hover:bg-emerald-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500"
                                            >
                                                Continue Shopping
                                            </button>
                                        </div>
                                    </div>
                                ) : (
                                    <div className="w-full h-[500px] sm:h-[600px] relative flex flex-col items-center justify-center">
                                        <p className="text-center text-gray-600 mb-4 font-medium text-sm sm:text-base px-2">
                                            Please complete the transaction in the window below.
                                        </p>

                                        {isIframeLoading && (
                                            <div className="absolute inset-0 bg-gray-50 flex flex-col items-center justify-center rounded-xl shadow-inner z-20">
                                                <p className="text-gray-700 font-medium text-sm">Loading secure payment portal...</p>
                                            </div>
                                        )}

                                        <iframe
                                            src={paystackAuthUrl}
                                            className={`w-full h-full border-2 border-gray-200 rounded-xl shadow-xl transition-opacity duration-500 ${isIframeLoading ? 'opacity-0' : 'opacity-100'}`}
                                            title="Payment Gateway"
                                            sandbox="allow-scripts allow-same-origin allow-forms allow-popups"
                                            onLoad={() => setIsIframeLoading(false)}
                                        ></iframe>
                                    </div>
                                )}
                            </div>
                        </motion.div>
                    </motion.div>
                )}
            </AnimatePresence>

            <div className="container mx-auto px-4 py-6 sm:py-8 max-w-7xl">
                {/* Header */}
                <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between mb-6 sm:mb-8 gap-4">
                    <div>
                        <h1 className="text-2xl sm:text-3xl font-bold text-gray-800">Shopping Cart</h1>
                        <p className="text-gray-600 mt-1">Review your items and proceed to checkout</p>
                    </div>
                    <a href={route('shop')} className="flex items-center text-green-600 hover:text-green-800 transition-colors self-start sm:self-auto">
                        <FaArrowLeft className="mr-2" />
                        Continue Shopping
                    </a>
                </div>

                {Object.values(cartItems).length > 0 ? (
                    <div className="flex flex-col lg:flex-row gap-6 lg:gap-8">
                        {/* Cart Items */}
                        <div className="flex-1 lg:flex-initial lg:w-2/3">
                            <div className="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                                <div className="px-4 sm:px-6 py-3 sm:py-4 bg-gray-50 border-b border-gray-200">
                                    <h2 className="text-base sm:text-lg font-semibold text-gray-800">Cart Items ({Object.values(cartItems).length})</h2>
                                </div>
                                
                                <div className="divide-y divide-gray-200">
                                    {Object.values(cartItems).map(item => (
                                        <div key={item.id} className="p-4 sm:p-6">
                                            <div className="flex gap-4">
                                                {/* Product Image */}
                                                <div className="w-20 h-20 sm:w-24 sm:h-24 bg-gray-100 rounded-lg overflow-hidden flex-shrink-0">
                                                    <img 
                                                        src={getFirstImage(item)} 
                                                        alt={item.name} 
                                                        className="w-full h-full object-cover"
                                                    />
                                                </div>
                                                
                                                {/* Product Details */}
                                                <div className="flex-1 min-w-0">
                                                    <div className="flex justify-between">
                                                        <div className="min-w-0 flex-1">
                                                            <h3 className="font-semibold text-gray-800 text-sm sm:text-lg truncate">{item.name}</h3>
                                                            <p className="text-gray-500 text-xs sm:text-sm mt-1">SKU: {item.attributes.sku || 'N/A'}</p>
                                                            {item.attributes.color && <p className="text-gray-500 text-xs sm:text-sm">Color: {item.attributes.color}</p>}
                                                            {item.attributes.size && <p className="text-gray-500 text-xs sm:text-sm">Size: {item.attributes.size}</p>}
                                                        </div>
                                                        <button 
                                                            onClick={() => handleRemoveItem(item.rowId)}
                                                            className="text-gray-400 hover:text-red-500 transition-colors flex-shrink-0 ml-2"
                                                            aria-label="Remove item"
                                                        >
                                                            <FaTrash className="text-sm sm:text-base" />
                                                        </button>
                                                    </div>
                                                    
                                                    <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between mt-3 sm:mt-4 gap-3">
                                                        {/* Price */}
                                                        <div className="font-semibold text-gray-800 text-sm sm:text-base">
                                                            ₵{item.price}
                                                        </div>
                                                        
                                                        {/* Quantity Controls */}
                                                        <div className="flex items-center border border-gray-300 rounded-md w-fit">
                                                            <button 
                                                                onClick={() => updateQuantity(item.rowId, item.quantity - 1)}
                                                                className="px-2 sm:px-3 py-1 sm:py-1 text-gray-600 hover:bg-gray-100 transition-colors"
                                                            >
                                                                <FaMinus className="text-xs" />
                                                            </button>
                                                            <span className="px-2 sm:px-4 py-1 border-x border-gray-300 text-sm sm:text-base">{item.quantity}</span>
                                                            <button 
                                                                onClick={() => updateQuantity(item.rowId, item.quantity + 1)}
                                                                className="px-2 sm:px-3 py-1 sm:py-1 text-gray-600 hover:bg-gray-100 transition-colors"
                                                            >
                                                                <FaPlus className="text-xs" />
                                                            </button>
                                                        </div>
                                                        
                                                        {/* Subtotal */}
                                                        <div className="font-semibold text-gray-800 text-sm sm:text-base">
                                                            ₵{(item.price * item.quantity).toFixed(2)}
                                                        </div>
                                                    </div>
                                                    
                                                    {/* Save for Later */}
                                                    <button className="mt-3 text-xs sm:text-sm text-green-600 hover:text-green-800 flex items-center">
                                                        <FaHeart className="mr-1" />
                                                        Save for Later
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            </div>
                            
                            {/* Trust Indicators */}
                            <div className="mt-6 grid grid-cols-1 sm:grid-cols-3 gap-3 sm:gap-4">
                                <div className="bg-white p-3 sm:p-4 rounded-lg shadow-sm border border-gray-200 flex items-center">
                                    <FaTruck className="text-green-600 mr-2 sm:mr-3 text-sm sm:text-base" />
                                    <div>
                                        <p className="font-semibold text-gray-800 text-xs sm:text-sm">Free Delivery</p>
                                        <p className="text-xs sm:text-sm text-gray-600">On orders over ₵50</p>
                                    </div>
                                </div>
                                <div className="bg-white p-3 sm:p-4 rounded-lg shadow-sm border border-gray-200 flex items-center">
                                    <FaUndo className="text-green-600 mr-2 sm:mr-3 text-sm sm:text-base" />
                                    <div>
                                        <p className="font-semibold text-gray-800 text-xs sm:text-sm">Easy Returns</p>
                                        <p className="text-xs sm:text-sm text-gray-600">30-day return policy</p>
                                    </div>
                                </div>
                                <div className="bg-white p-3 sm:p-4 rounded-lg shadow-sm border border-gray-200 flex items-center">
                                    <FaLock className="text-green-600 mr-2 sm:mr-3 text-sm sm:text-base" />
                                    <div>
                                        <p className="font-semibold text-gray-800 text-xs sm:text-sm">Secure Payment</p>
                                        <p className="text-xs sm:text-sm text-gray-600">SSL encrypted</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        {/* Order Summary */}
                        <div className="flex-1 lg:flex-initial lg:w-1/3">
                            <div className="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden sticky top-4">
                                <div className="px-4 sm:px-6 py-3 sm:py-4 bg-gray-50 border-b border-gray-200">
                                    <h2 className="text-base sm:text-lg font-semibold text-gray-800">Order Summary</h2>
                                </div>
                                
                                <div className="p-4 sm:p-6">
                                    <div className="space-y-3">
                                        <div className="flex justify-between">
                                            <span className="text-gray-600 text-sm sm:text-base">Subtotal</span>
                                            <span className="font-medium text-sm sm:text-base">₵{getSubtotal().toFixed(2)}</span>
                                        </div>
                                        
                                        {discount > 0 && (
                                            <div className="flex justify-between text-green-600">
                                                <span className="text-sm sm:text-base">Discount</span>
                                                <span className="font-medium text-sm sm:text-base">-₵{discount.toFixed(2)}</span>
                                            </div>
                                        )}
                                        
                                        <div className="flex justify-between">
                                            <span className="text-gray-600 text-sm sm:text-base">Shipping</span>
                                            <span className="font-medium text-green-600 text-sm sm:text-base">Free</span>
                                        </div>
                                        
                                        <div className="flex justify-between">
                                            <span className="text-gray-600 text-sm sm:text-base">Tax</span>
                                            <span className="font-medium text-sm sm:text-base">₵{(getSubtotal() * 0.08).toFixed(2)}</span>
                                        </div>
                                        
                                        <div className="border-t pt-3">
                                            <div className="flex justify-between font-bold text-base sm:text-lg">
                                                <span>Total</span>
                                                <span>₵{getTotalWithDiscountAndTax()}</span>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    {/* Coupon Code */}
                                    <div className="mt-6">
                                        <div className="flex">
                                            <div className="relative flex-grow">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <FaTag className="text-gray-400 text-sm" />
                                                </div>
                                                <input 
                                                    type="text" 
                                                    value={couponCode}
                                                    onChange={(e) => setCouponCode(e.target.value)}
                                                    placeholder="Coupon code" 
                                                    className="w-full pl-10 pr-3 py-2 border border-gray-300 rounded-l-md focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-transparent text-sm"
                                                />
                                            </div>
                                            <button 
                                                onClick={applyCoupon}
                                                className="bg-gray-100 hover:bg-gray-200 text-gray-800 px-3 sm:px-4 py-2 rounded-r-md transition-colors text-sm"
                                            >
                                                Apply
                                            </button>
                                        </div>
                                    </div>
                                    
                                    <div className="mt-6">
                                        <h3 className="text-base sm:text-lg font-semibold text-gray-800 mb-4">Checkout Details</h3>
                                        <div className="space-y-4">
                                            <div>
                                                <label htmlFor="name" className="block text-sm font-medium text-gray-700">Full Name</label>
                                                <input type="text" id="name" name="name" value={formData.name} onChange={handleInputChange} className="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500 text-sm" />
                                            </div>
                                            <div>
                                                <label htmlFor="email" className="block text-sm font-medium text-gray-700">Email Address</label>
                                                <input type="email" id="email" name="email" value={formData.email} onChange={handleInputChange} className="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500 text-sm" />
                                            </div>
                                            <div>
                                                <label htmlFor="address" className="block text-sm font-medium text-gray-700">Shipping Address</label>
                                                <textarea id="address" name="address" rows="3" value={formData.address} onChange={handleInputChange} className="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500 text-sm"></textarea>
                                            </div>
                                        </div>
                                    </div>

                                    {/* Payment Method Selection */}
                                    <div className="mt-6">
                                        <h3 className="text-base sm:text-lg font-semibold text-gray-800 mb-4">Payment Method</h3>
                                        <div className="space-y-3">
                                            <div className="flex items-center">
                                                <input id="paystack" name="paymentMethod" type="radio" value="paystack" checked={paymentMethod === 'paystack'} onChange={(e) => setPaymentMethod(e.target.value)} className="focus:ring-green-500 h-4 w-4 text-green-600 border-gray-300" />
                                                <label htmlFor="paystack" className="ml-3 block text-sm font-medium text-gray-700">
                                                    Paystack
                                                </label>
                                            </div>
                                            <div className="flex items-center">
                                                <input id="paypal" name="paymentMethod" type="radio" value="paypal" checked={paymentMethod === 'paypal'} onChange={(e) => setPaymentMethod(e.target.value)} className="focus:ring-green-500 h-4 w-4 text-green-600 border-gray-300" />
                                                <label htmlFor="paypal" className="ml-3 block text-sm font-medium text-gray-700">
                                                    PayPal
                                                </label>
                                            </div>
                                        </div>
                                    </div>

                                    {/* Checkout Button */}
                                    <button 
                                        onClick={handleCheckout}
                                        disabled={!isFormValid() || isProcessing}
                                        className="w-full bg-green-600 hover:bg-green-700 text-white font-medium py-3 px-4 rounded-md mt-6 transition-colors flex items-center justify-center disabled:bg-gray-400 disabled:cursor-not-allowed text-sm sm:text-base"
                                    >
                                        {isProcessing ? (
                                            <>
                                                <svg className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                                    <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                                </svg>
                                                Processing...
                                            </>
                                        ) : (
                                            <>
                                                <FaShoppingBag className="mr-2" />
                                                Proceed to Checkout
                                            </>
                                        )}
                                    </button>                                    
                                    
                                    {/* Payment Methods */}
                                    <div className="mt-6">
                                        <p className="text-sm text-gray-600 mb-2">We accept:</p>
                                        <div className="flex space-x-4">
                                            <div className="flex items-center space-x-1">
                                                <FaCreditCard className="text-gray-600 text-lg sm:text-xl" />
                                                <span className="text-xs sm:text-sm text-gray-700">Paystack</span>
                                            </div>
                                            <div className="flex items-center space-x-1">
                                                <FaCcPaypal className="text-blue-700 text-lg sm:text-xl" />
                                                <span className="text-xs sm:text-sm text-gray-700">PayPal</span>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    {/* Help */}
                                    <div className="mt-6 text-center">
                                        <p className="text-sm text-gray-600">Need help? <a href="#" className="text-green-600 hover:text-green-800">Contact Support</a></p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                ) : (
                    <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-8 sm:p-12 text-center">
                        <div className="w-20 h-20 sm:w-24 sm:h-24 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-6">
                            <FaShoppingBag className="text-3xl sm:text-4xl text-gray-400" />
                        </div>
                        <h2 className="text-xl sm:text-2xl font-semibold text-gray-800 mb-2">Your cart is empty</h2>
                        <p className="text-gray-600 mb-6 text-sm sm:text-base">Looks like you haven't added any items to your cart yet.</p>
                        <a 
                            href={route('shop')} 
                            className="inline-flex items-center bg-green-600 hover:bg-green-700 text-white font-medium py-2 px-4 sm:py-2 sm:px-6 rounded-md transition-colors text-sm sm:text-base"
                        >
                            Start Shopping
                        </a>
                    </div>
                )}
            </div>
        </HomeLayout>
    );
}