import React, { useEffect, useState } from 'react';
import { Head } from '@inertiajs/react';
import Navbar from '@/Components/Navbar';
import Footer from '@/Components/Footer';
import { motion } from 'framer-motion';
import Preloader from '@/Components/Preloader';

const containerVariants = {
    hidden: { opacity: 0 },
    visible: {
        opacity: 1,
        transition: {
            staggerChildren: 0.15,
            delayChildren: 0.3
        }
    }
};

const itemVariants = {
    hidden: { y: 20, opacity: 0 },
    visible: {
        y: 0,
        opacity: 1,
        transition: { duration: 0.5, ease: "easeOut" }
    }
};

const PortfolioCard = ({ item }) => {
    const vibrantGradient = "bg-gradient-to-br from-yellow-500 to-red-600";

    return (
        <motion.div
            className="group relative overflow-hidden rounded-2xl shadow-lg border border-gray-100 transition-all duration-300 cursor-pointer"
            variants={itemVariants}
            whileHover={{
                y: -5,
                boxShadow: "0 20px 25px -5px rgba(251, 191, 36, 0.4), 0 10px 10px -5px rgba(251, 191, 36, 0.1)"
            }}
            whileTap={{ scale: 0.98 }}
        >
            <div className="relative overflow-hidden">
                <img
                    src={item.image_url}
                    alt={item.title}
                    className="w-full object-contain transform group-hover:scale-105 transition-transform duration-500"
                    onError={(e) => {
                        e.target.onerror = null;
                        e.target.src = "https://placehold.co/600x400/FECACA/7F1D1D?text=Project"
                    }}
                />

                <div className={`absolute inset-0 p-8 flex flex-col justify-end transition-opacity duration-500 ${vibrantGradient} opacity-0 group-hover:opacity-95`}>
                    <h3 className="text-2xl font-bold text-white mb-2">{item.title}</h3>
                    <p className="text-yellow-100 text-base mb-4 italic">{item.description}</p>
                </div>
            </div>

            <div className="p-6 bg-white">
                <h3 className="text-xl font-bold text-gray-900 mb-3">{item.title}</h3>
                <p className="text-gray-600 text-sm">{item.description}</p>
                <div className="mt-4">
                    <span className="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-red-100 text-red-800">
                        View Project
                    </span>
                </div>
            </div>
        </motion.div>
    );
};

const Portfolio = ({ portfolios, setting }) => {
    const [loading, setLoading] = useState(true);
        
    useEffect(() => {
        const timer = setTimeout(() => {
            setLoading(false);
        }, 1000);

        return () => clearTimeout(timer);
    }, []);

    if (loading) {
        return <Preloader />;
    }

    return (
        <div className="min-h-screen bg-white mt-24 font-sans antialiased">
            <Head title="Our Portfolio" />
            <Navbar setting={setting} />

            {/* Professional Hero Section - Matching the provided structure */}
            <div className="relative bg-gradient-to-r from-red-700 to-red-800 py-16 text-white">
                <div className="container mx-auto px-6">
                    <div className="max-w-3xl">
                        <h1 className="text-4xl md:text-5xl font-bold mb-4">Our Portfolio</h1>
                        <p className="text-xl text-red-100 max-w-2xl">
                            Explore our showcase of successful projects and discover how we've helped businesses achieve their goals through innovative solutions.
                        </p>
                    </div>
                </div>
                
                {/* Subtle decorative elements */}
                <div className="absolute top-0 right-0 w-96 h-96 bg-yellow-400 rounded-full mix-blend-soft-light filter blur-3xl opacity-20"></div>
            </div>

            <section className="py-20 lg:py-32 bg-gradient-to-b from-white to-yellow-50" id="portfolio">
                <div className="container mx-auto px-6">
                    {/* Section Header */}
                    <motion.div
                        className="text-center mb-16"
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.7 }}
                        viewport={{ once: true }}
                    >
                        <h2 className="text-3xl font-bold text-gray-800 mb-4">Featured Projects</h2>
                        <p className="text-lg text-gray-600 max-w-2xl mx-auto">
                            A selection of our recent work that demonstrates our expertise and commitment to excellence.
                        </p>
                    </motion.div>

                    {/* Portfolio Grid */}
                    <motion.div
                        className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8"
                        variants={containerVariants}
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true, amount: 0.1 }}
                    >
                        {portfolios && portfolios.length > 0 ? (
                            portfolios.map((item) => (
                                <PortfolioCard key={item.id} item={item} />
                            ))
                        ) : (
                            <div className="col-span-full text-center py-10">
                                <p className="text-xl text-gray-600">No portfolios available at the moment.</p>
                            </div>
                        )}
                    </motion.div>

                    {/* Call to Action Section */}
                    <motion.div
                        className="mt-20 bg-gradient-to-r from-red-700 to-red-800 rounded-2xl p-10 md:p-16 text-white shadow-xl overflow-hidden"
                        initial={{ opacity: 0, scale: 0.95 }}
                        whileInView={{ opacity: 1, scale: 1 }}
                        transition={{ duration: 0.7 }}
                        viewport={{ once: true }}
                    >
                        {/* Decorative elements */}
                        <div className="absolute top-0 right-0 w-64 h-64 bg-yellow-400 rounded-full mix-blend-soft-light filter blur-3xl opacity-20"></div>
                        <div className="absolute bottom-0 left-0 w-64 h-64 bg-yellow-400 rounded-full mix-blend-soft-light filter blur-3xl opacity-20"></div>
                        
                        <div className="relative z-10 text-center">
                            <h3 className="text-3xl font-bold mb-4">Interested in Working Together?</h3>
                            <p className="text-red-100 max-w-2xl mx-auto mb-8">
                                Let's discuss how we can bring your vision to life. Our team is ready to create a solution that exceeds your expectations.
                            </p>
                            <button className="px-8 py-3 bg-gradient-to-r from-yellow-500 to-yellow-600 text-white font-bold rounded-lg shadow-lg hover:from-yellow-600 hover:to-yellow-700 transition-all duration-300 transform hover:scale-105">
                                Start Your Project
                            </button>
                        </div>
                    </motion.div>
                </div>
            </section>

            <Footer setting={setting} />
        </div>
    );
};

export default Portfolio;