import React, { useState } from 'react';
import { Head } from '@inertiajs/react';
import HomeLayout from '@/Layouts/HomeLayout';
import BookingModal from '@/Components/BookingModal';
import ServiceCard from '@/Components/ServiceCard';

export default function Services({ services, setting }) {
    console.log('Services Page: services prop', services);
    const [showBookingModal, setShowBookingModal] = useState(false);
    const [selectedService, setSelectedService] = useState(null);
    const [activeCategory, setActiveCategory] = useState('all');

    // Service categories for filtering
    const categories = [
        { id: 'all', name: 'All Services', icon: '🌿' },
        { id: 'consulting', name: 'Consulting', icon: '💼' },
        { id: 'training', name: 'Training', icon: '📚' },
        { id: 'support', name: 'Support', icon: '🛠️' },
        { id: 'implementation', name: 'Implementation', icon: '🚀' }
    ];

    const openBookingModal = (service) => {
        setSelectedService(service);
        setShowBookingModal(true);
    };

    const closeBookingModal = () => {
        setShowBookingModal(false);
        setSelectedService(null);
    };

    const handleOpenModal = (service) => {
        setSelectedService(service);
        setShowBookingModal(true);
    };

    // Filter services based on category
    const filteredServices = activeCategory === 'all' 
        ? services 
        : services.filter(service => service.category === activeCategory);

    console.log('Services Page: filteredServices', filteredServices);

    return (
        <HomeLayout>
            <Head title="Services" />
            
            {/* Hero Section with Green Gradient */}
            <div className="relative bg-gradient-to-br from-emerald-900 via-green-800 to-teal-900 overflow-hidden">
                {/* Decorative Elements */}
                <div className="absolute inset-0 bg-black opacity-20"></div>
                <div className="absolute top-0 right-0 w-96 h-96 bg-emerald-500 rounded-full filter blur-3xl opacity-10 -translate-y-1/2 translate-x-1/2"></div>
                <div className="absolute bottom-0 left-0 w-96 h-96 bg-teal-500 rounded-full filter blur-3xl opacity-10 translate-y-1/2 -translate-x-1/2"></div>
                
                <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12 sm:py-16 md:py-24 text-center">
                    <h1 className="text-3xl sm:text-4xl md:text-5xl lg:text-6xl font-extrabold tracking-tight text-white mb-4 sm:mb-6">
                        Our Premium Services
                    </h1>
                    <p className="text-lg sm:text-xl md:text-2xl text-green-100 max-w-3xl mx-auto mb-6 sm:mb-8 px-4">
                        Discover our comprehensive range of solutions designed to elevate your business to new heights
                    </p>
                </div>
            </div>
            
            {/* Services Section */}
            <div className="py-12 sm:py-16 bg-gray-50">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6 sm:gap-8">
                        {filteredServices.map((service, index) => (
                            <ServiceCard key={service.id}
                                    service={service}
                                    onBookService={handleOpenModal} />
                        ))}
                    </div>
                    
                    {filteredServices.length === 0 && (
                        <div className="text-center py-12 sm:py-16 bg-white rounded-lg shadow-sm">
                            <div className="mx-auto w-20 h-20 sm:w-24 sm:h-24 bg-gray-100 rounded-full flex items-center justify-center mb-4">
                                <svg className="h-10 w-10 sm:h-12 sm:w-12 text-gray-400" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.172 16.172a4 4 0 015.656 0M9 10h.01M15 10h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                </svg>
                            </div>
                            <h3 className="text-lg sm:text-xl font-medium text-gray-900 mb-2">No services found</h3>
                            <p className="text-gray-500 max-w-md mx-auto px-4">
                                We couldn't find any services in this category. Please try another category.
                            </p>
                            <button 
                                onClick={() => setActiveCategory('all')}
                                className="mt-6 px-6 py-2 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700 transition-colors"
                            >
                                View All Services
                            </button>
                        </div>
                    )}
                </div>
            </div>
            
            {/* Process Section */}
            <div className="py-12 sm:py-16 bg-white">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="text-center mb-8 sm:mb-12">
                        <h2 className="text-2xl sm:text-3xl font-bold text-gray-800 mb-3 sm:mb-4">How It Works</h2>
                        <p className="text-base sm:text-lg text-gray-600 max-w-2xl mx-auto px-4">
                            Our simple 4-step process to get you started with our services
                        </p>
                    </div>
                    
                    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 sm:gap-8">
                        {[
                            { step: 1, title: 'Choose Service', description: 'Browse our catalog and select the service that best fits your needs' },
                            { step: 2, title: 'Book Appointment', description: 'Select a convenient time slot and book your appointment' },
                            { step: 3, title: 'Consultation', description: 'Meet with our experts to discuss your requirements in detail' },
                            { step: 4, title: 'Get Results', description: 'Receive your customized solution and start seeing the benefits' }
                        ].map((item) => (
                            <div key={item.step} className="text-center">
                                <div className="mx-auto w-14 h-14 sm:w-16 sm:h-16 bg-emerald-100 rounded-full flex items-center justify-center mb-3 sm:mb-4">
                                    <span className="text-xl sm:text-2xl font-bold text-emerald-600">{item.step}</span>
                                </div>
                                <h3 className="text-base sm:text-lg font-semibold text-gray-800 mb-2">{item.title}</h3>
                                <p className="text-sm sm:text-base text-gray-600 px-2">{item.description}</p>
                            </div>
                        ))}
                    </div>
                </div>
            </div>
            
            {/* CTA Section */}
            <div className="bg-gradient-to-r from-emerald-600 to-teal-700 py-12 sm:py-16">
                <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                    <h2 className="text-2xl sm:text-3xl font-bold text-white mb-3 sm:mb-4">
                        Need a Custom Service?
                    </h2>
                    <p className="text-lg sm:text-xl text-green-100 mb-6 sm:mb-8 max-w-2xl mx-auto px-4">
                        Our team can create tailored solutions to meet your unique business requirements.
                    </p>
                    <button className="px-6 sm:px-8 py-2 sm:py-3 bg-white text-emerald-600 font-medium rounded-lg hover:bg-gray-100 transition-colors shadow-lg">
                        Contact Us
                    </button>
                </div>
            </div>

            {selectedService && (
                <BookingModal
                    service={selectedService}
                    show={showBookingModal}
                    onClose={closeBookingModal}
                />
            )}
        </HomeLayout>
    );
}