import React, { useState } from 'react';
import HomeLayout from '@/Layouts/HomeLayout';
import ProductCard from '@/Components/ProductCard';
import { router } from '@inertiajs/react';

const Shop = ({ products }) => {
    const [activeCategory, setActiveCategory] = useState('all');
    const [searchTerm, setSearchTerm] = useState('');
    
    // Categories for filtering
    const categories = [
        { id: 'all', name: 'All Products', icon: '📦' },
        { id: 'software', name: 'Software Solutions', icon: '💻' },
        { id: 'consulting', name: 'Consulting Services', icon: '👔' },
        { id: 'training', name: 'Training Programs', icon: '📚' },
        { id: 'infrastructure', name: 'Infrastructure', icon: '🏢' }
    ];
    
    const handleProductClick = (product) => {
        router.post(route('cart.add'), {
            product_id: product.id,
            quantity: 1, // Assuming 1 for now, can be made dynamic later
        }, {
            onSuccess: () => {
                alert(`${product.name} added to cart successfully!`);
            },
            onError: (errors) => {
                alert(`Failed to add ${product.name} to cart: ${Object.values(errors).join(', ')}`);
            },
            preserveScroll: true,
        });
    };
    
    // Filter products based on category and search
    const filteredProducts = products && products.length > 0 
        ? products.filter(product => {
            const matchesCategory = activeCategory === 'all' || product.category === activeCategory;
            const matchesSearch = product.name.toLowerCase().includes(searchTerm.toLowerCase()) || 
                                 product.description.toLowerCase().includes(searchTerm.toLowerCase());
            return matchesCategory && matchesSearch;
        })
        : [];

    return (
        <HomeLayout title="Corporate Solutions">
            {/* Hero Section with Green Gradient Background */}
            <div className="relative bg-gradient-to-br from-emerald-900 via-green-800 to-teal-900 overflow-hidden">
                {/* Decorative Elements */}
                <div className="absolute inset-0 bg-black opacity-20"></div>
                <div className="absolute top-0 right-0 w-96 h-96 bg-emerald-500 rounded-full filter blur-3xl opacity-10 -translate-y-1/2 translate-x-1/2"></div>
                <div className="absolute bottom-0 left-0 w-96 h-96 bg-teal-500 rounded-full filter blur-3xl opacity-10 translate-y-1/2 -translate-x-1/2"></div>
                
                <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-24 text-center">
                    <h1 className="text-5xl md:text-6xl font-extrabold tracking-tight text-white mb-6">
                        Premium Corporate Solutions
                    </h1>
                    <p className="text-xl md:text-2xl text-green-100 max-w-3xl mx-auto mb-8">
                        Elevate your business with our comprehensive suite of enterprise-grade products and services
                    </p>
                </div>
            </div>
            
            {/* Search and Filter Section */}
            <div className="bg-white shadow-sm border-b border-gray-100 sticky top-0 z-10">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
                    <div className="flex flex-col md:flex-row gap-4 items-center justify-between">
                        {/* Search Bar */}
                        <div className="relative w-full md:w-96">
                            <input
                                type="text"
                                placeholder="Search products..."
                                className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-transparent transition-all"
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                            />
                            <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <svg className="h-5 w-5 text-gray-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                    <path fillRule="evenodd" d="M8 4a4 4 0 100 8 4 4 0 000-8zM2 8a6 6 0 1110.89 3.476l4.817 4.817a1 1 0 01-1.414 1.414l-4.816-4.816A6 6 0 012 8z" clipRule="evenodd" />
                                </svg>
                            </div>
                        </div>
                        
                        {/* Category Filter */}
                        <div className="flex flex-wrap gap-2">
                            {categories.map(category => (
                                <button
                                    key={category.id}
                                    onClick={() => setActiveCategory(category.id)}
                                    className={`px-4 py-2 rounded-full text-sm font-medium transition-all ${
                                        activeCategory === category.id
                                            ? 'bg-emerald-600 text-white shadow-md'
                                            : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                                    }`}
                                >
                                    <span className="mr-2">{category.icon}</span>
                                    {category.name}
                                </button>
                            ))}
                        </div>
                    </div>
                </div>
            </div>
            
            {/* Main Content */}
            <div className="py-16 bg-gray-50">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Results Count */}
                    <div className="mb-8 flex justify-between items-center">
                        <h2 className="text-2xl font-bold text-gray-800">
                            {activeCategory === 'all' ? 'All Solutions' : categories.find(c => c.id === activeCategory)?.name}
                        </h2>
                        <p className="text-gray-600">
                            Showing {filteredProducts.length} {filteredProducts.length === 1 ? 'product' : 'products'}
                        </p>
                    </div>
                    
                    {/* Product Grid */}
                    {filteredProducts.length > 0 ? (
                        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-8">
                            {filteredProducts.map((product, index) => (
                                <div 
                                    key={product.id} 
                                    className="transform transition-all duration-300 hover:-translate-y-2"
                                    style={{ animationDelay: `${index * 50}ms` }}
                                >
                                    <ProductCard 
                                        product={product} 
                                        onProductClick={handleProductClick} 
                                        className="h-full shadow-lg hover:shadow-xl transition-shadow duration-300"
                                    />
                                </div>
                            ))}
                        </div>
                    ) : (
                        <div className="text-center py-16 bg-white rounded-lg shadow-sm">
                            <div className="mx-auto w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mb-4">
                                <svg className="h-12 w-12 text-gray-400" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.172 16.172a4 4 0 015.656 0M9 10h.01M15 10h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                </svg>
                            </div>
                            <h3 className="text-xl font-medium text-gray-900 mb-2">No products found</h3>
                            <p className="text-gray-500 max-w-md mx-auto">
                                We couldn't find any products matching your criteria. Try adjusting your filters or search terms.
                            </p>
                            <button 
                                onClick={() => {setActiveCategory('all'); setSearchTerm('');}}
                                className="mt-6 px-6 py-2 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700 transition-colors"
                            >
                                Reset Filters
                            </button>
                        </div>
                    )}
                </div>
            </div>
            
            {/* CTA Section with Green Gradient */}
            <div className="bg-gradient-to-r from-emerald-600 to-teal-700 py-16">
                <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                    <h2 className="text-3xl font-bold text-white mb-4">
                        Need a Custom Solution?
                    </h2>
                    <p className="text-xl text-green-100 mb-8 max-w-2xl mx-auto">
                        Our team of experts can develop tailored solutions to meet your specific business requirements.
                    </p>
                    <button className="px-8 py-3 bg-white text-emerald-600 font-medium rounded-lg hover:bg-gray-100 transition-colors shadow-lg">
                        Request a Consultation
                    </button>
                </div>
            </div>
        </HomeLayout>
    );
};

export default Shop;