import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link } from '@inertiajs/react';
import { useState, useEffect } from 'react';
import { format } from 'date-fns';
import Pagination from '@/Components/Pagination';
import Modal from '@/Components/Modal';
import axios from 'axios';
import { 
    CalendarIcon, 
    ClockIcon, 
    UserIcon, 
    CreditCardIcon,
    CheckCircleIcon as ConfirmedIcon,
    ExclamationCircleIcon as AlertIcon,
    InformationCircleIcon as InfoIcon,
    MagnifyingGlassIcon,
    XMarkIcon,
    FunnelIcon,
    ChevronDownIcon,
    CurrencyDollarIcon,
    AcademicCapIcon
} from '@heroicons/react/24/outline';
import { CheckIcon, XCircleIcon } from '@heroicons/react/24/solid';

// Helper component for Booking Status Icon
const StatusIcon = ({ status }) => {
    switch (status) {
        case 'confirmed':
            return <ConfirmedIcon className="h-4 w-4" />;
        case 'pending':
            return <ClockIcon className="h-4 w-4" />;
        case 'completed':
            return <CheckIcon className="h-4 w-4" />;
        case 'cancelled':
            return <XCircleIcon className="h-4 w-4" />;
        default:
            return <InfoIcon className="h-4 w-4" />;
    }
};

// Helper component for Booking Status Color
const getStatusColor = (status) => {
    switch (status) {
        case 'confirmed':
            return 'bg-green-100 text-green-800 border-green-200';
        case 'pending':
            return 'bg-yellow-100 text-yellow-800 border-yellow-200';
        case 'completed':
            return 'bg-blue-100 text-blue-800 border-blue-200';
        case 'cancelled':
            return 'bg-red-100 text-red-800 border-red-200';
        default:
            return 'bg-gray-100 text-gray-800 border-gray-200';
    }
};

export default function Index({ auth, schoolServiceBookings }) {
    const [searchTerm, setSearchTerm] = useState('');
    const [filterStatus, setFilterStatus] = useState('all');
    const [filteredSchoolServiceBookings, setFilteredSchoolServiceBookings] = useState(schoolServiceBookings?.data || []);
    const [activeTab, setActiveTab] = useState('upcoming');
    const [showModal, setShowModal] = useState(false);
    const [selectedBooking, setSelectedBooking] = useState(null);
    const [isFilterExpanded, setIsFilterExpanded] = useState(false);
    const [isMobileView, setIsMobileView] = useState(false);

    // Detect screen size
    useEffect(() => {
        const handleResize = () => {
            setIsMobileView(window.innerWidth < 768);
        };
        
        handleResize(); // Initial check
        window.addEventListener('resize', handleResize);
        
        return () => window.removeEventListener('resize', handleResize);
    }, []);

    useEffect(() => {
        let filtered = schoolServiceBookings?.data || [];
        const today = new Date();
        today.setHours(0, 0, 0, 0); // Normalize today to start of day for accurate comparison

        // Filter by search term
        if (searchTerm) {
            filtered = filtered.filter(booking =>
                booking.school_service_name.toLowerCase().includes(searchTerm.toLowerCase())
            );
        }

        // Filter by status
        if (filterStatus !== 'all') {
            filtered = filtered.filter(booking => booking.status === filterStatus);
        }

        // Filter by tab (upcoming/past/all)
        if (activeTab === 'upcoming') {
            filtered = filtered.filter(booking => new Date(booking.booking_date) >= today);
        } else if (activeTab === 'past') {
            filtered = filtered.filter(booking => new Date(booking.booking_date) < today);
        }
        // 'all' tab uses filtered data as is.

        setFilteredSchoolServiceBookings(filtered);
    }, [searchTerm, filterStatus, activeTab, schoolServiceBookings?.data]);

    const viewDetails = (booking) => {
        axios.get(route('user.school-services.show', booking.id))
            .then((response) => {
                setSelectedBooking(response.data.booking);
                setShowModal(true);
            })
            .catch(error => {
                console.error("Error fetching booking details:", error);
            });
    };

    const closeModal = () => {
        setShowModal(false);
        setSelectedBooking(null);
    };

    const StatCard = ({ title, value, icon, bgColor }) => (
        <div className="bg-white overflow-hidden shadow-lg rounded-xl border border-gray-100 hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1">
            <div className="p-3 sm:p-5">
                <div className="flex items-center">
                    <div className={`${bgColor} rounded-xl p-2 sm:p-3 flex-shrink-0`}>
                        {icon}
                    </div>
                    <div className="ml-3 sm:ml-5 w-0 flex-1">
                        <dl>
                            <dt className="text-xs sm:text-sm font-medium text-gray-500 truncate">{title}</dt>
                            <dd className="text-lg sm:text-xl font-bold text-gray-900">{value}</dd>
                        </dl>
                    </div>
                </div>
            </div>
        </div>
    );
    
    // Calculate stats for cards
    const totalBookings = schoolServiceBookings?.total || 0;
    const confirmedCount = schoolServiceBookings?.data ? schoolServiceBookings.data.filter(b => b.status === 'confirmed').length : 0;
    const pendingCount = schoolServiceBookings?.data ? schoolServiceBookings.data.filter(b => b.status === 'pending').length : 0;
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    const upcomingCount = schoolServiceBookings?.data ? schoolServiceBookings.data.filter(b => new Date(b.booking_date) >= today).length : 0;
    const totalSpent = schoolServiceBookings?.data ? schoolServiceBookings.data.reduce((sum, booking) => sum + parseFloat(booking.total_price || 0), 0).toFixed(2) : '0.00';

    return (
        <AdminLayout>
            <Head title="My School Service Bookings" />

            <div className="py-4 sm:py-6 bg-gray-50 min-h-screen">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    
                    {/* Header Section with Gradient Background */}
                    <div className="bg-gradient-to-r from-green-600 to-emerald-700 rounded-xl shadow-xl p-4 sm:p-6 mb-6 sm:mb-8 text-white">
                        <h1 className="text-2xl sm:text-3xl font-bold mb-2">School Service Bookings 🏫</h1>
                        <p className="text-green-100 text-sm sm:text-base">Welcome back, {auth.user.name}. Here you can manage all your school service bookings.</p>
                    </div>

                    {/* Stats Cards with Enhanced Design */}
                    <div className="grid grid-cols-2 sm:grid-cols-2 lg:grid-cols-4 gap-3 sm:gap-4 mb-6 sm:mb-8">
                        <StatCard 
                            title="Total Bookings" 
                            value={totalBookings} 
                            icon={<CalendarIcon className="h-5 w-5 sm:h-6 sm:w-6 text-white" />} 
                            bgColor="bg-green-600" 
                        />
                        <StatCard 
                            title="Confirmed" 
                            value={confirmedCount} 
                            icon={<ConfirmedIcon className="h-5 w-5 sm:h-6 sm:w-6 text-white" />} 
                            bgColor="bg-green-500" 
                        />
                        <StatCard 
                            title="Pending" 
                            value={pendingCount} 
                            icon={<ClockIcon className="h-5 w-5 sm:h-6 sm:w-6 text-white" />} 
                            bgColor="bg-yellow-600" 
                        />
                        <StatCard 
                            title="Total Spent" 
                            value={`GH₵${totalSpent}`} 
                            icon={<CurrencyDollarIcon className="h-5 w-5 sm:h-6 sm:w-6 text-white" />} 
                            bgColor="bg-emerald-600" 
                        />
                    </div>

                    {/* Main Content Card */}
                    <div className="bg-white shadow-xl rounded-xl overflow-hidden border border-gray-100">
                        <div className="p-4 sm:p-6 border-b border-gray-100">
                            <h2 className="text-xl font-semibold text-gray-800">Booking Management</h2>
                        </div>
                        
                        {/* Search, Filter, and Tabs Section */}
                        <div className="p-4 sm:p-6">
                            <div className="flex flex-col gap-3 sm:gap-4">
                                <div className="relative w-full">
                                    <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                        <MagnifyingGlassIcon className="h-5 w-5 text-gray-400" />
                                    </div>
                                    <input
                                        type="text"
                                        className="block w-full pl-10 pr-3 py-2 sm:py-3 border border-gray-300 rounded-lg leading-5 bg-white placeholder-gray-500 focus:outline-none focus:placeholder-gray-400 focus:ring-2 focus:ring-green-500 focus:border-green-500 text-sm transition"
                                        placeholder="Search by service name..."
                                        value={searchTerm}
                                        onChange={(e) => setSearchTerm(e.target.value)}
                                    />
                                </div>
                                <div className="flex justify-between items-center">
                                    <button
                                        onClick={() => setIsFilterExpanded(!isFilterExpanded)}
                                        className="inline-flex items-center px-3 py-2 border border-gray-300 shadow-sm text-sm leading-4 font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                    >
                                        <FunnelIcon className="h-4 w-4 mr-2" />
                                        Filters
                                        <ChevronDownIcon className={`ml-2 h-4 w-4 transform transition-transform ${isFilterExpanded ? 'rotate-180' : ''}`} />
                                    </button>
                                </div>
                            </div>
                            
                            {isFilterExpanded && (
                                <div className="mt-4 pt-4 border-t border-gray-200">
                                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-1">Status</label>
                                            <select
                                                className="block w-full px-3 py-2 border border-gray-300 bg-white rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500 text-sm"
                                                value={filterStatus}
                                                onChange={(e) => setFilterStatus(e.target.value)}
                                            >
                                                <option value="all">All Statuses</option>
                                                <option value="confirmed">Confirmed</option>
                                                <option value="pending">Pending</option>
                                                <option value="completed">Completed</option>
                                                <option value="cancelled">Cancelled</option>
                                            </select>
                                        </div>
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-1">Date Range</label>
                                            <select
                                                className="block w-full px-3 py-2 border border-gray-300 bg-white rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500 text-sm"
                                            >
                                                <option value="all">All Time</option>
                                                <option value="today">Today</option>
                                                <option value="week">This Week</option>
                                                <option value="month">This Month</option>
                                                <option value="year">This Year</option>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                            )}

                            {/* Tabs - Responsive */}
                            <div className="border-b border-gray-200 overflow-x-auto mt-4">
                                <nav className="-mb-px flex space-x-2 sm:space-x-8 min-w-max">
                                    {['upcoming', 'past', 'all'].map((tab) => (
                                        <button
                                            key={tab}
                                            onClick={() => setActiveTab(tab)}
                                            className={`py-2 px-1 sm:px-1 border-b-2 font-medium text-xs sm:text-sm capitalize whitespace-nowrap transition-colors duration-200 ${
                                                activeTab === tab
                                                    ? 'border-green-600 text-green-600 font-semibold'
                                                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                            }`}
                                        >
                                            {tab} ({schoolServiceBookings?.data ? schoolServiceBookings.data.filter(b => {
                                                const bookingDate = new Date(b.booking_date);
                                                if (tab === 'upcoming') return bookingDate >= today;
                                                if (tab === 'past') return bookingDate < today;
                                                return true;
                                            }).length : 0})
                                        </button>
                                    ))}
                                </nav>
                            </div>
                        </div>

                        {/* Bookings List with Enhanced Design - Responsive */}
                        <div className="bg-white shadow-xl rounded-xl overflow-hidden border border-gray-100">
                            {filteredSchoolServiceBookings.length > 0 ? (
                                <>
                                    {/* Desktop Table View */}
                                    <div className="hidden md:block overflow-x-auto">
                                        <table className="min-w-full divide-y divide-gray-200">
                                            <thead className="bg-gray-50">
                                                <tr>
                                                    <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Service Details
                                                    </th>
                                                    <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Date
                                                    </th>
                                                    <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Quantity
                                                    </th>
                                                    <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Total Price
                                                    </th>
                                                    <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Status
                                                    </th>
                                                    <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Actions
                                                    </th>
                                                </tr>
                                            </thead>
                                            <tbody className="bg-white divide-y divide-gray-200">
                                                {filteredSchoolServiceBookings.map((booking) => (
                                                    <tr key={booking.id} className="hover:bg-gray-50 transition-colors">
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="flex items-center">
                                                                <div className="flex-shrink-0 h-10 w-10 bg-green-100 rounded-full flex items-center justify-center">
                                                                    <AcademicCapIcon className="h-6 w-6 text-green-600" />
                                                                </div>
                                                                <div className="ml-4">
                                                                    <div className="text-sm font-medium text-gray-900">{booking.school_service_name}</div>
                                                                </div>
                                                            </div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="text-sm text-gray-900">{format(new Date(booking.booking_date), 'MMM dd, yyyy')}</div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="text-sm text-gray-900">{booking.quantity || 1}</div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="text-sm font-medium text-gray-900">GH₵{parseFloat(booking.total_price || 0).toFixed(2)}</div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusColor(booking.status)}`}>
                                                                <StatusIcon status={booking.status} />
                                                                <span className="ml-1">{booking.status ? booking.status.charAt(0).toUpperCase() + booking.status.slice(1) : 'Unknown'}</span>
                                                            </span>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                                            <button 
                                                                onClick={() => viewDetails(booking)}
                                                                className="text-green-600 hover:text-green-900"
                                                            >
                                                                View Details
                                                            </button>
                                                        </td>
                                                    </tr>
                                                ))}
                                            </tbody>
                                        </table>
                                    </div>

                                    {/* Mobile Card View */}
                                    <div className="md:hidden">
                                        <ul className="divide-y divide-gray-200">
                                            {filteredSchoolServiceBookings.map((booking) => (
                                                <li key={booking.id} className="p-4 hover:bg-gray-50 transition-colors">
                                                    <div className="flex items-start justify-between">
                                                        <div className="flex items-center">
                                                            <div className="flex-shrink-0 h-10 w-10 bg-green-100 rounded-full flex items-center justify-center">
                                                                <AcademicCapIcon className="h-6 w-6 text-green-600" />
                                                            </div>
                                                            <div className="ml-3">
                                                                <p className="text-sm font-medium text-gray-900">{booking.school_service_name}</p>
                                                                <p className="text-xs text-gray-500">{format(new Date(booking.booking_date), 'MMM dd, yyyy')}</p>
                                                            </div>
                                                        </div>
                                                        <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusColor(booking.status)}`}>
                                                            <StatusIcon status={booking.status} />
                                                        </span>
                                                    </div>
                                                    <div className="mt-3 grid grid-cols-2 gap-2 text-sm">
                                                        <div>
                                                            <span className="text-gray-500">Qty:</span>
                                                            <span className="ml-1 font-medium">{booking.quantity || 1}</span>
                                                        </div>
                                                        <div>
                                                            <span className="text-gray-500">Total:</span>
                                                            <span className="ml-1 font-medium">GH₵{parseFloat(booking.total_price || 0).toFixed(2)}</span>
                                                        </div>
                                                    </div>
                                                    <div className="mt-3 flex justify-end">
                                                        <button 
                                                            onClick={() => viewDetails(booking)}
                                                            className="text-green-600 hover:text-green-900 text-sm font-medium"
                                                        >
                                                            View Details
                                                        </button>
                                                    </div>
                                                </li>
                                            ))}
                                        </ul>
                                    </div>
                                </>
                            ) : (
                                <div className="text-center py-12">
                                    <AcademicCapIcon className="mx-auto h-12 w-12 text-gray-400" />
                                    <h3 className="mt-2 text-sm font-medium text-gray-900">No bookings found</h3>
                                    <p className="mt-1 text-sm text-gray-500">
                                        Try adjusting your search term, status filter, or tab selection.
                                    </p>
                                </div>
                            )}
                        </div>
                        
                        {/* Pagination */}
                        <div className="p-4 sm:p-6 border-t border-gray-100">
                            <Pagination 
                                links={schoolServiceBookings?.links} 
                                from={schoolServiceBookings?.from} 
                                to={schoolServiceBookings?.to} 
                                total={schoolServiceBookings?.total} 
                            />
                        </div>
                    </div>
                </div>
            </div>

            {/* Enhanced Modal with Modern Design - Responsive */}
            <Modal show={showModal} onClose={closeModal} maxWidth="lg">
                <div className="bg-white rounded-xl shadow-2xl overflow-hidden max-h-[90vh] overflow-y-auto">
                    {/* Modal Header with Gradient Background */}
                    <div className="bg-gradient-to-r from-green-600 to-emerald-700 px-4 sm:px-6 py-3 sm:py-4">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center">
                                <div className="flex-shrink-0 bg-white/20 p-2 rounded-lg backdrop-blur-sm">
                                    <AcademicCapIcon className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                                </div>
                                <div className="ml-3 sm:ml-4">
                                    <h2 className="text-lg sm:text-xl font-bold text-white">Booking Details</h2>
                                    <p className="text-green-100 text-xs sm:text-sm">Reference: #{selectedBooking?.id}</p>
                                </div>
                            </div>
                            <button
                                type="button"
                                className="text-white/80 hover:text-white transition-colors p-1 rounded-md hover:bg-white/10"
                                onClick={closeModal}
                            >
                                <XMarkIcon className="h-5 w-5 sm:h-6 sm:w-6" />
                            </button>
                        </div>
                    </div>

                    {/* Modal Body */}
                    <div className="px-4 sm:px-6 py-4 sm:py-5">
                        {selectedBooking ? (
                            <div className="space-y-4 sm:space-y-6">
                                {/* Status Badge */}
                                <div className="bg-gray-50 rounded-lg p-3 sm:p-4">
                                    <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between mb-3 sm:mb-4">
                                        <h3 className="text-base font-semibold text-gray-900 mb-2 sm:mb-0">Current Status</h3>
                                        <span className={`inline-flex items-center px-3 py-1 rounded-full text-sm font-medium border ${getStatusColor(selectedBooking.status)}`}>
                                            <StatusIcon status={selectedBooking.status} />
                                            <span className="ml-2">{selectedBooking.status ? selectedBooking.status.charAt(0).toUpperCase() + selectedBooking.status.slice(1) : 'Unknown'}</span>
                                        </span>
                                    </div>
                                    
                                    {/* Timeline */}
                                    <div className="relative">
                                        <div className="absolute inset-0 flex items-center" aria-hidden="true">
                                            <div className="w-full border-t border-gray-300"></div>
                                        </div>
                                        <div className="relative flex justify-between">
                                            {[
                                                { step: 'Booked', completed: true, icon: CalendarIcon },
                                                { step: 'Confirmed', completed: selectedBooking.status !== 'pending', icon: ConfirmedIcon },
                                                { step: 'Completed', completed: selectedBooking.status === 'completed', icon: CheckIcon }
                                            ].map((step, index) => (
                                                <div key={index} className="flex flex-col items-center">
                                                    <div className={`flex items-center justify-center w-6 h-6 sm:w-8 sm:h-8 rounded-full border-2 ${
                                                        step.completed 
                                                            ? 'bg-green-600 border-green-600' 
                                                            : 'bg-white border-gray-300'
                                                    }`}>
                                                        {step.completed ? (
                                                            <step.icon className="w-3 h-3 sm:w-4 sm:h-4 text-white" />
                                                        ) : (
                                                            <div className="w-1.5 h-1.5 sm:w-2 sm:h-2 bg-gray-400 rounded-full"></div>
                                                        )}
                                                    </div>
                                                    <span className={`mt-1 sm:mt-2 text-xs font-medium ${
                                                        step.completed 
                                                            ? 'text-gray-900' 
                                                            : 'text-gray-500'
                                                    }`}>
                                                        {step.step}
                                                    </span>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                </div>

                                {/* Booking Information Grid */}
                                <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 sm:gap-6">
                                    <div className="bg-white border border-gray-200 rounded-lg p-3 sm:p-4">
                                        <h3 className="text-sm font-semibold text-gray-900 mb-2 sm:mb-3 flex items-center">
                                            <AcademicCapIcon className="h-4 w-4 sm:h-5 sm:w-5 mr-2 text-green-600" />
                                            Service Information
                                        </h3>
                                        <dl className="space-y-1.5 sm:space-y-2">
                                            <div className="flex justify-between">
                                                <dt className="text-xs sm:text-sm text-gray-500">Service:</dt>
                                                <dd className="text-xs sm:text-sm font-medium text-gray-900">{selectedBooking.school_service_name}</dd>
                                            </div>
                                            <div className="flex justify-between">
                                                <dt className="text-xs sm:text-sm text-gray-500">Booking Date:</dt>
                                                <dd className="text-xs sm:text-sm font-medium text-gray-900">{format(new Date(selectedBooking.booking_date), 'MMM dd, yyyy')}</dd>
                                            </div>
                                            <div className="flex justify-between">
                                                <dt className="text-xs sm:text-sm text-gray-500">Quantity:</dt>
                                                <dd className="text-xs sm:text-sm font-medium text-gray-900">{selectedBooking.quantity || 1}</dd>
                                            </div>
                                        </dl>
                                    </div>

                                    <div className="bg-white border border-gray-200 rounded-lg p-3 sm:p-4">
                                        <h3 className="text-sm font-semibold text-gray-900 mb-2 sm:mb-3 flex items-center">
                                            <CurrencyDollarIcon className="h-4 w-4 sm:h-5 sm:w-5 mr-2 text-green-600" />
                                            Price Details
                                        </h3>
                                        <dl className="space-y-1.5 sm:space-y-2">
                                            <div className="flex justify-between">
                                                <dt className="text-xs sm:text-sm text-gray-500">Service Price:</dt>
                                                <dd className="text-xs sm:text-sm font-medium text-gray-900">GH₵{parseFloat(selectedBooking.total_price || 0).toFixed(2)}</dd>
                                            </div>
                                            <div className="flex justify-between">
                                                <dt className="text-xs sm:text-sm text-gray-500">Processing Fee:</dt>
                                                <dd className="text-xs sm:text-sm font-medium text-gray-900">GH₵0.00</dd>
                                            </div>
                                            <div className="flex justify-between pt-1.5 sm:pt-2 border-t border-gray-200">
                                                <dt className="text-xs sm:text-sm font-semibold text-gray-900">Total:</dt>
                                                <dd className="text-sm sm:text-base font-bold text-green-600">GH₵{parseFloat(selectedBooking.total_price || 0).toFixed(2)}</dd>
                                            </div>
                                        </dl>
                                    </div>
                                </div>

                                {/* Customer Information */}
                                <div className="bg-white border border-gray-200 rounded-lg p-3 sm:p-4">
                                    <h3 className="text-sm font-semibold text-gray-900 mb-2 sm:mb-3 flex items-center">
                                        <UserIcon className="h-4 w-4 sm:h-5 sm:w-5 mr-2 text-green-600" />
                                        Customer Information
                                    </h3>
                                    <dl className="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4">
                                        <div>
                                            <dt className="text-xs sm:text-sm text-gray-500">Name:</dt>
                                            <dd className="text-xs sm:text-sm font-medium text-gray-900">{auth.user.name}</dd>
                                        </div>
                                        <div>
                                            <dt className="text-xs sm:text-sm text-gray-500">Email:</dt>
                                            <dd className="text-xs sm:text-sm font-medium text-gray-900">{auth.user.email}</dd>
                                        </div>
                                    </dl>
                                </div>

                                {/* Notes Section */}
                                {selectedBooking.notes && (
                                    <div className="bg-gray-100 rounded-lg p-3 sm:p-4">
                                        <h3 className="text-sm font-semibold text-gray-900 mb-2 sm:mb-3 flex items-center">
                                            <InfoIcon className="h-4 w-4 sm:h-5 sm:w-5 mr-2 text-gray-500" />
                                            Additional Notes
                                        </h3>
                                        <p className="text-xs sm:text-sm text-gray-700 italic border-l-4 border-gray-300 pl-3 py-1">"{selectedBooking.notes}"</p>
                                    </div>
                                )}

                                {/* Action Buttons */}
                                <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center pt-3 sm:pt-4 border-t border-gray-200">
                                    <div className="flex flex-col sm:flex-row space-y-2 sm:space-y-0 sm:space-x-2 mb-3 sm:mb-0">
                                        {selectedBooking.status === 'pending' && (
                                            <button className="w-full sm:w-auto inline-flex justify-center items-center px-3 sm:px-4 py-2 border border-red-300 rounded-md shadow-sm text-xs sm:text-sm font-medium text-red-700 bg-white hover:bg-red-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500">
                                                <XCircleIcon className="w-4 h-4 mr-2" />
                                                Cancel Booking
                                            </button>
                                        )}
                                    </div>
                                    <button
                                        type="button"
                                        className="w-full sm:w-auto px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 transition-colors text-sm font-medium"
                                        onClick={closeModal}
                                    >
                                        Close
                                    </button>
                                </div>
                            </div>
                        ) : (
                            <div className="text-center py-8">
                                <InfoIcon className="mx-auto h-10 w-10 text-gray-400" />
                                <p className="mt-2 text-sm text-gray-500">Unable to load booking details</p>
                            </div>
                        )}
                    </div>
                </div>
            </Modal>
        </AdminLayout>
    );
}